<?php
// kyc.php — live search + ajax pagination
session_start();
error_reporting(0);
require_once 'config.php';

// -----------------------
// Inputs
// -----------------------
$q            = trim($_GET['q'] ?? '');
$records_page = 10;
$page         = max(1, (int)($_GET['page'] ?? 1));

// -----------------------
// WHERE + params (aspnetusers assumed)
// Columns used: fullname, email, country, kyc_verify, reg_date
// If your column names differ, adjust below queries.
// -----------------------
$whereParts = ['COALESCE(kyc_verify,1) = 1'];   // ← always pending
$params     = [];

if ($q !== '') {
  $params[':q'] = '%'.$q.'%';
  $whereParts[] = "("
                . "COALESCE(NULLIF(TRIM(fullname),''), '') LIKE :q "
                . "OR email LIKE :q "
                . "OR COALESCE(NULLIF(TRIM(country),''), '') LIKE :q"
                . ")";
}

$whereSql = 'WHERE '.implode(' AND ', $whereParts);
 

// -----------------------
// Count
// -----------------------
$sqlCount = "SELECT COUNT(*) FROM aspnetusers $whereSql";
$st = $conn->prepare($sqlCount);
foreach ($params as $k=>$v) $st->bindValue($k, $v, PDO::PARAM_STR);
$st->execute();
$total = (int)$st->fetchColumn();

$total_pages = max(1, (int)ceil($total / $records_page));
$page        = min($page, $total_pages);
$offset      = ($page - 1) * $records_page;

// -----------------------
// Page data
// -----------------------
$sql = "SELECT 
          COALESCE(NULLIF(TRIM(fullname),''), email) AS fullname,
          email,
          COALESCE(NULLIF(TRIM(country),''), '—') AS country,
          COALESCE(kyc_verify, 0) AS kyc_verify,
          COALESCE(reg_date, '—') AS reg_date
        FROM aspnetusers
        $whereSql
        ORDER BY reg_date DESC, email ASC
        LIMIT :offset, :limit";
$st = $conn->prepare($sql);
foreach ($params as $k=>$v) $st->bindValue($k, $v, PDO::PARAM_STR);
$st->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
$st->bindValue(':limit',  (int)$records_page, PDO::PARAM_INT);
$st->execute();
$rows = $st->fetchAll(PDO::FETCH_ASSOC);

// -----------------------
// Render helpers
// -----------------------
function kycPill($v){
  $v = (int)$v;
  if ($v === 1) {
    return '<span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-semibold bg-emerald-50 text-emerald-700 border border-emerald-200">Approved</span>';
  }
  return '<span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-semibold bg-amber-50 text-amber-700 border border-amber-200">Pending</span>';
}
function render_rows($rows){
  ob_start();
  if ($rows) {
    foreach ($rows as $r) {
      echo '<tr>';
      echo '<td>'.htmlentities($r['fullname']).'</td>';
      echo '<td>'.htmlentities($r['email']).'</td>';
      echo '<td>'.htmlentities($r['country']).'</td>';
      echo '<td>'.kycPill($r['kyc_verify']).'</td>';
      echo '<td>'.htmlentities($r['reg_date']).'</td>';
      echo '</tr>';
    }
  } else {
    echo '<tr><td colspan="5" style="text-align:center; color:#6b7280; padding:18px;">No KYC records to display.</td></tr>';
  }
  return ob_get_clean();
}
function page_link($page, $label, $isActive=false, $isDisabled=false){
  $classes = 'page-item';
  if ($isActive)   $classes .= ' active';
  if ($isDisabled) $classes .= ' disabled';
  $aria = htmlspecialchars((string)$label, ENT_QUOTES);
  // data-page triggers AJAX
  echo '<li class="'.$classes.'"><a class="page-link js-page" href="#" data-page="'.(int)$page.'" aria-label="'.$aria.'">'.$label.'</a></li>';
}
function render_pagination($page, $total_pages){
  ob_start();
  $window = 2;
  $start  = max(1, $page - $window);
  $end    = min($total_pages, $page + $window);
  echo '<nav aria-label="Page navigation example"><ul class="pagination">';
  page_link(1, '&laquo;&laquo; First', false, $page===1);
  page_link(max(1,$page-1), '&laquo; Prev', false, $page===1);
  if ($start > 1) {
    page_link(1, '1', $page===1);
    if ($start > 2) echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
  }
  for ($i=$start; $i<=$end; $i++) page_link($i, (string)$i, $i===$page);
  if ($end < $total_pages) {
    if ($end < $total_pages - 1) echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
    page_link($total_pages, (string)$total_pages, $page===$total_pages);
  }
  page_link(min($total_pages,$page+1), 'Next &raquo;', false, $page===$total_pages);
  page_link($total_pages, 'Last &raquo;&raquo;', false, $page===$total_pages);
  echo '</ul></nav>';
  return ob_get_clean();
}

// -----------------------
// AJAX: return JSON with tbody + pagination + meta
// -----------------------
if (isset($_GET['ajax']) && $_GET['ajax'] === 'list') {
  $from = $total ? ($offset + 1) : 0;
  $to   = min($offset + $records_page, $total);
  $resp = [
    'tbody'      => render_rows($rows),
    'pagination' => render_pagination($page, $total_pages),
    'meta'       => [
      'from'  => $from,
      'to'    => $to,
      'total' => $total,
    ],
  ];
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($resp);
  exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1"/>
  <title>KYC</title>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css"/>
  <style>
    :root{
      --vm-bg:#f6f7fb; --vm-white:#fff; --vm-text:#1f2937; --vm-muted:#6b7280;
      --vm-orange:#ff7a1a; --vm-orange-600:#e86500; --vm-orange-100:#fff3e8;
      --vm-border:#e5e7eb; --vm-shadow:0 8px 24px rgba(0,0,0,.07);
    }
    body{ margin:0; background:var(--vm-bg); color:var(--vm-text); font-family:system-ui,-apple-system,Segoe UI,Roboto,Ubuntu,Cantarell,'Helvetica Neue',Arial; }
    @media (min-width:768px){ main.md\:ml-64{ margin-left:16rem!important; } }
    .dashboard-content-one{ max-width:1200px; margin:0 auto; padding:24px 18px 40px; }
    .card{ border:0; border-radius:18px; background:var(--vm-white); box-shadow:var(--vm-shadow); overflow:hidden; }
    .card-body{ padding:18px; }
    .table{ width:100%; border-collapse:separate; border-spacing:0; }
    .table thead th{ position:sticky; top:0; z-index:1; padding:12px 14px; text-align:left; color:#fff; background:linear-gradient(180deg,var(--vm-orange),var(--vm-orange-600)); }
    .table tbody td{ padding:12px 14px; border-bottom:1px solid var(--vm-border); }
    .table tbody tr:nth-child(even){ background:#fffdfb; }
    .table tbody tr:hover{ background:var(--vm-orange-100); }
    .table-responsive{ border:1px solid var(--vm-border); border-radius:14px; overflow:auto; }
    .pagination{ display:flex; gap:.5rem; justify-content:center; flex-wrap:wrap; margin:18px 0 6px; padding:0; list-style:none; }
    .page-item .page-link{ display:inline-flex; align-items:center; justify-content:center; min-width:40px; height:40px; padding:0 .9rem; border:1px solid var(--vm-border); border-radius:12px; background:#fff; color:var(--vm-text); font-weight:600; text-decoration:none; transition:.18s; }
    .page-item .page-link:hover{ border-color:var(--vm-orange); color:var(--vm-orange); box-shadow:0 6px 14px rgba(255,122,26,.15); transform:translateY(-1px); }
    .page-item.active .page-link{ background:linear-gradient(135deg,var(--vm-orange),var(--vm-orange-600)); color:#fff; border-color:transparent; box-shadow:0 6px 14px rgba(232,101,0,.25); }
    .page-item.disabled .page-link{ opacity:.45; pointer-events:none; background:#fafafa; }
    .search-wrap{ display:flex; align-items:center; gap:.5rem; flex-wrap:wrap; margin:8px 0 16px; }
    .search-input{ flex:1 1 260px; border:1px solid var(--vm-border); border-radius:12px; padding:.6rem .9rem; background:#fff; color:var(--vm-text); outline:none; }
    .search-input:focus{ border-color:var(--vm-orange); box-shadow:0 0 0 3px rgba(255,122,26,.15); }
    .reset-btn{ border-radius:12px; border:1px solid var(--vm-border); background:#fff; color:var(--vm-text); padding:.6rem .9rem; font-weight:600; text-decoration:none; }
    .search-meta{ color:var(--vm-muted); font-size:.9rem; margin:-4px 0 8px; }
    .fade-enter{ opacity:.35; }
    /* pills already styled inline in kycPill() */
  </style>
</head>
<body>
<?php include 'header.php'; ?>
<div class="flex flex-col md:flex-row h-screen">
  <?php include 'side_bar.php'; ?>

  <main class="flex-1 p-4 md:p-6 md:ml-64">
    <div class="dashboard-content-one">
      <h2 style="margin:4px 0 12px; font-size:1.4rem; font-weight:700; color:#1f2937;">KYC</h2>

      <!-- Live search -->
      <div class="search-wrap">
        <input id="liveSearch" type="text"
               value="<?= htmlspecialchars($q, ENT_QUOTES) ?>"
               placeholder="Search by Name, Email, Country, or type 'approved' / 'pending'..."
               class="search-input" autocomplete="off">
        <?php if ($q !== ''): ?>
          <a class="reset-btn" href="?">Reset</a>
        <?php endif; ?>
      </div>
      <?php $from = $total ? ($offset + 1) : 0; $to = min($offset + $records_page, $total); ?>
      <div id="searchMeta" class="search-meta">
        <?= $q !== '' ? 'Showing results for “'.htmlspecialchars($q, ENT_QUOTES).'”. ' : '' ?>
        <?= $total ? "Showing <strong>$from–$to</strong> of <strong>$total</strong>." : "No records found." ?>
      </div>

      <div class="card">
        <div class="card-body">
          <div id="tableWrap" class="table-responsive">
            <table class="table">
              <thead>
                <tr>
                  <th>Name</th>
                  <th>Email</th>
                  <th>Country</th>
                  <th>KYC</th>
                  <th>Registered</th>
                </tr>
              </thead>
              <tbody id="tbody">
                <?= render_rows($rows) ?>
              </tbody>
            </table>
          </div>

          <div id="paginationWrap">
            <?= render_pagination($page, $total_pages) ?>
          </div>
        </div>
      </div>
    </div>
  </main>
</div>
<?php include 'footer.php'; ?>

<script>
// Debounce helper
const debounce = (fn, ms=300) => { let t; return (...a)=>{ clearTimeout(t); t=setTimeout(()=>fn(...a), ms); }; };

const input          = document.getElementById('liveSearch');
const tbody          = document.getElementById('tbody');
const paginationWrap = document.getElementById('paginationWrap');
const meta           = document.getElementById('searchMeta');
const tableWrap      = document.getElementById('tableWrap');

async function loadList(page = 1){
  const q = input.value.trim();
  tableWrap.classList.add('fade-enter');
  try {
    const url = new URL(window.location.href);
    url.searchParams.set('ajax', 'list');
    url.searchParams.set('page', page);
    url.searchParams.set('q', q);
    const res  = await fetch(url.toString(), { credentials:'same-origin' });
    const data = await res.json();

    tbody.innerHTML          = data.tbody || '';
    paginationWrap.innerHTML = data.pagination || '';

    const from  = data?.meta?.from ?? 0;
    const to    = data?.meta?.to ?? 0;
    const total = data?.meta?.total ?? 0;
    meta.innerHTML = (q ? `Showing results for “${escapeHtml(q)}”. ` : '') +
                     (total ? `Showing <strong>${from}–${to}</strong> of <strong>${total}</strong>.` : 'No records found.');

    bindPagination();
  } catch(err) {
    console.error(err);
  } finally {
    tableWrap.classList.remove('fade-enter');
  }
}

function bindPagination(){
  paginationWrap.querySelectorAll('a.js-page').forEach(a => {
    a.addEventListener('click', (ev) => {
      ev.preventDefault();
      const p = parseInt(a.dataset.page || '1', 10);
      if (!isNaN(p)) loadList(p);
    }, { once:true });
  });
}

function escapeHtml(s){
  const d = document.createElement('div'); d.innerText = s; return d.innerHTML;
}

// type → search
input.addEventListener('input', debounce(() => loadList(1), 250));

// first bind for initial HTML
bindPagination();
</script>
</body>
</html>
