<?php
@session_start();
require_once __DIR__ . '/config.php';

if (empty($_SESSION['clogin'])) {
  echo "<script>window.location.href='login.php';</script>"; exit;
}
if (!isset($conn) || !($conn instanceof PDO)) {
  die('Database connection not established. Check config.php (PDO).');
}

/* ---------- PDO helpers (parameterized) ---------- */
function scalarp(PDO $conn, string $sql, array $params = []) {
  $st = $conn->prepare($sql);
  $st->execute($params);
  return $st->fetchColumn();
}
function rowsp(PDO $conn, string $sql, array $params = []): array {
  $st = $conn->prepare($sql);
  $st->execute($params);
  return $st->fetchAll(PDO::FETCH_ASSOC);
}

$me = $_SESSION['clogin'];

/* ---------- Metrics (scoped to logged-in email) ---------- */
$totUsers      = (int) scalarp($conn, "SELECT COUNT(*) FROM aspnetusers WHERE email = :e", [':e'=>$me]);
$verifiedUsers = (int) scalarp($conn, "SELECT COUNT(*) FROM aspnetusers WHERE email = :e AND email_confirmed = 1", [':e'=>$me]);

/* If kyc_update has an email column (common case): */
$kycApproved   = (int) scalarp($conn, "SELECT COUNT(*) FROM kyc_update WHERE email = :e AND Status = 1", [':e'=>$me]);
$kycPending    = (int) scalarp($conn, "SELECT COUNT(*) FROM kyc_update WHERE email = :e AND Status = 0", [':e'=>$me]);

/* If liveaccount has an email column (typical MT5 mirror tables do): */
$liveAccounts  = (int) scalarp($conn, "SELECT COUNT(*) FROM liveaccount WHERE email = :e", [':e'=>$me]);
$sumBalance    = (float) (scalarp($conn, "SELECT IFNULL(SUM(deposit_amount),0) FROM total_balance WHERE email = :e", [':e'=>$me]) ?? 0);
$sumwithdraw  = (float) (scalarp($conn, "SELECT IFNULL(SUM(withdraw_amount),0) FROM total_balance WHERE email = :e", [':e'=>$me]) ?? 0);
$sumEquity     = (float) (scalarp($conn, "SELECT IFNULL(SUM(equity),0) FROM liveaccount WHERE email = :e", [':e'=>$me]) ?? 0);

/* Wallet totals (user-scoped) */
$walletTrend = rowsp($conn, "
  SELECT
    ROUND(COALESCE(SUM(deposit_amount),  0), 2) AS dep,
    ROUND(COALESCE(SUM(withdraw_amount), 0), 2) AS wdr
  FROM total_balance
  WHERE email = :e
", [':e'=>$me]);

$wt_dep = array_map('floatval', array_column($walletTrend, 'dep'));
$wt_wdr = array_map('floatval', array_column($walletTrend, 'wdr'));

/* Top accounts for this user */
$topAccounts = rowsp($conn, "
  SELECT trade_id, account_name, Balance, server, equity
  FROM liveaccount
  WHERE email = :e
  ORDER BY equity DESC
  LIMIT 5
", [':e'=>$me]);

/* “Recent users” scoped to this user = just show this user’s row */
$recentUsers = rowsp($conn, "
  SELECT fullname, email, country, kyc_verify, client_status, reg_date
  FROM aspnetusers
  WHERE email = :e
  LIMIT 1
", [':e'=>$me]);

/* Leverage breakdown (only if your leverage table stores email) */
try {
  $levCounts = rowsp($conn, "
    SELECT account_type, COUNT(*) AS cnt
    FROM leverage
    WHERE email = :e
    GROUP BY account_type
    ORDER BY cnt DESC
  ", [':e'=>$me]);
} catch (Throwable $t) {
  // Fallback if leverage table has no email column:
  $levCounts = [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Dashboard</title>
  <!-- TailwindCSS -->
  <script src="https://cdn.tailwindcss.com"></script>
  <script>
    tailwind.config = {
      theme: {
        extend: {
          colors: {
            brand: {
              50:'#eef7ff',100:'#d9edff',200:'#bfe0ff',300:'#94caff',400:'#63adff',
              500:'#3a90ff',600:'#2977e6',700:'#1f5fbe',800:'#1b4e98',900:'#183f79'
            },
            accent: {
              50:'#fff7ed',100:'#ffedd5',200:'#fed7aa',300:'#fdba74',400:'#fb923c',
              500:'#f97316',600:'#ea580c',700:'#c2410c',800:'#9a3412',900:'#7c2d12'
            }
          },
          boxShadow: { soft: '0 10px 25px rgba(0,0,0,.08)' },
          borderRadius: { xl2: '1.25rem' }
        }
      }
    }
  </script>
  <!-- Chart.js + DataTables -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/jquery.dataTables.min.css" />
  <script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
  <!-- Icons -->
  <link href="https://cdnjs.cloudflare.com/ajax/libs/remixicon/4.2.0/remixicon.min.css" rel="stylesheet"/>
  <style>
    .card{ box-shadow: var(--tw-shadow,0 10px 25px rgba(0,0,0,.08)); }
  </style>
</head>
<body class="min-h-screen bg-gradient-to-b from-brand-50 to-white text-gray-900">

<?php include __DIR__ . '/header.php'; ?>
<div class="flex flex-col md:flex-row">
  <?php include __DIR__ . '/side_bar.php'; ?>

  <main class="flex-1 p-4 sm:p-6 md:ml-64">
    <div class="max-w-7xl mx-auto">

      <!-- Page heading -->
      <div class="mb-4 sm:mb-6 flex items-center justify-between">
        <!--<div>-->
        <!--  <h1 class="text-2xl sm:text-3xl font-semibold tracking-tight">Demo ComPany</h1>-->
        <!--  <p class="text-sm text-gray-600">Live overview of users, KYC, accounts, and funds.</p>-->
        <!--</div>-->
        <!--<button id="themeToggle" class="hidden sm:inline-flex items-center gap-2 px-3 py-2 rounded-xl bg-brand-600 hover:bg-brand-700 text-white text-sm"><i class="ri-sun-fill"></i><span>Theme</span></button>-->
      </div>

      <!-- KPI Tiles -->
<!-- Remix Icon (for ri-* icons you already use) -->
<link href="https://cdn.jsdelivr.net/npm/remixicon@4.3.0/fonts/remixicon.css" rel="stylesheet">

<style>
  :root{
    --orange:#F1671B;
    --blue:#0aa0e0;
    --green:#1AA049;
    --red:#C03221;
    --card:#ffffff;
    --ink:#0f172a;
    --muted:#64748b;
    --ring:rgba(241,103,27,.18);
    --shadow:0 10px 22px rgba(15,23,42,.06);
    --radius:10px;
  }

  /* Grid like your screenshot: 2/3/6 cols responsively */
  .stats-grid{
    display:grid; gap:14px;
    grid-template-columns:repeat(1,minmax(0,1fr));
  }
  @media (min-width: 768px){ .stats-grid{ grid-template-columns:repeat(3,minmax(0,1fr)); } }
  @media (min-width: 1280px){ .stats-grid{ grid-template-columns:repeat(6,minmax(0,1fr)); } }

  /* Card */
  .stat-card{
    position:relative;
    background:var(--card);
    border:1px solid var(--ring);
    border-radius:var(--radius);
    padding:8px;
    box-shadow:var(--shadow);
  }
  .stat-card::before{
    content:""; position:absolute; left:0; right:0; top:0; height:6px;
    background:var(--accent, var(--orange));
    border-top-left-radius:var(--radius); border-top-right-radius:var(--radius);
  }
  .stat-head{ display:flex; align-items:center; gap:8px; }
  .icon-chip{
    width:26px; height:26px; border-radius:14px;
    background:var(--orange); color:#111827;
    display:grid; place-items:center; box-shadow:inset 0 -2px 0 rgba(0,0,0,.08);
  }
  .icon-chip i{ font-size:22px; }

  .stat-label{ color:#475569; font-weight:700; }
  .stat-value{ color:#0b2540; font-size:14px; font-weight:400; line-height:1.1; margin-top:6px; }
  .stat-sub{ font-size:12px; margin-top:4px; font-weight:300; }

  /* Top bar theme colors per tile */
  .is-blue  { --accent: var(--blue); }
  .is-orange{ --accent: var(--orange); }
  .is-green { --accent: var(--green); }
  .is-red   { --accent: var(--red); }

  /* Helpful tint colors for subs (like your image) */
  .sub-green{ color:#059669; }
  .sub-amber{ color:#b45309; }
  .sub-gray { color:#64748b; }
</style>
<!-- OVERRIDES: bigger cards + 3 columns on desktop (=> 2 rows for 6 tiles) -->
<style>
  /* Optional: widen the area a bit */
  .stats-wide{max-width:1400px;margin:0 auto}

  /* Grid: 1 col mobile → 2 cols tablet → 3 cols desktop */
  .stats-grid{display:grid;gap:18px;grid-template-columns:repeat(1,minmax(0,1fr))}
  @media (min-width:768px){ .stats-grid{ grid-template-columns:repeat(2,minmax(0,1fr)); } }
  @media (min-width:1200px){ .stats-grid{ grid-template-columns:repeat(3,minmax(0,1fr)); } }

  /* Bigger cards */
  .stat-card{ padding:22px 22px 18px; border-radius:22px; }
  .stat-card::before{ height:10px; }

  /* Larger icon chip + text */
  .icon-chip{ width:52px;height:52px;border-radius:16px; }
  .icon-chip i{ font-size:24px; }
  .stat-label{ font-size:15px; }
  .stat-value{ font-size:14px; }
  .stat-sub{ font-size:13px; }
</style>

<div class="stats-wide">
  <section class="stats-grid">
    <!-- Total Users -->
    <div class="stat-card is-blue">
      <div class="stat-head">
        <div class="icon-chip"><i class="ri-team-fill"></i></div>
        <div class="stat-label">Total Users</div>
      </div>
      <div class="stat-value"><?php echo number_format($totUsers); ?></div>
      <div class="stat-sub sub-green">Verified: <?php echo number_format($verifiedUsers); ?></div>
    </div>

    <!-- KYC Status -->
    <div class="stat-card is-orange">
      <div class="stat-head">
        <div class="icon-chip"><i class="ri-shield-check-fill"></i></div>
        <div class="stat-label">KYC Status</div>
      </div>
      <div class="stat-value">
        <?php echo number_format($kycApproved); ?>
        <span style="font-size:14px;color:#94a3b8;font-weight:700">approved</span>
      </div>
      <div class="stat-sub sub-amber">Pending: <?php echo number_format($kycPending); ?></div>
    </div>

    <!-- Live Accounts -->
    <div class="stat-card is-green">
      <div class="stat-head">
        <div class="icon-chip"><i class="ri-account-circle-fill"></i></div>
        <div class="stat-label">Live Accounts</div>
      </div>
      <div class="stat-value"><?php echo number_format($liveAccounts); ?></div>
      <div class="stat-sub sub-gray">Platform: MetaTrader5</div>
    </div>

    <!-- Balance (Σ) -->
    <div class="stat-card is-red">
      <div class="stat-head">
        <div class="icon-chip"><i class="ri-wallet-3-fill"></i></div>
        <div class="stat-label">Deposits (Σ)</div>
      </div>
      <div class="stat-value">$<?php echo number_format($sumBalance,2); ?></div>
      <div class="stat-sub sub-gray">Equity: $<?php echo number_format($sumEquity,2); ?></div>
    </div>

    <!-- Deposits (Σ) -->
    <div class="stat-card is-blue">
      <div class="stat-head">
        <div class="icon-chip"><i class="ri-arrow-down-circle-fill"></i></div>
        <div class="stat-label">Withdrawals (Σ)</div>
      </div>
      <div class="stat-value">$<?php echo number_format($sumwithdraw,2); ?></div>
      <div class="stat-sub sub-gray">Withdrawals: $</div>
    </div>

    <!-- Leverage Options -->
    <div class="stat-card is-green">
      <div class="stat-head">
        <div class="icon-chip"><i class="ri-slideshow-3-fill"></i></div>
        <div class="stat-label">Leverage Options</div>
      </div>
      <div class="stat-value"><?php echo count($levCounts); ?></div>
      <div class="stat-sub sub-gray">Types configured</div>
    </div>
  </section>
</div>


      <!-- Charts Row -->
<!-- Remix Icon (icons) -->
<link href="https://cdn.jsdelivr.net/npm/remixicon@4.3.0/fonts/remixicon.css" rel="stylesheet">
<!-- Chart.js for the charts -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

<?php
  /* Safe PHP helpers (use your real values) */
  $dep  = (float)($totalDeposit  ?? 0);
  $wdr  = (float)($totalWithdraw ?? 0);
  $net  = $dep - $wdr;
  $sum  = max(0.000001, $dep + $wdr);
  $depPct = round($dep / $sum * 100, 2);
  $wdrPct = round($wdr / $sum * 100, 2);

  $kycApproved = (int)($kycApproved ?? 0);
  $kycPending  = (int)($kycPending  ?? 0);
  $kycRejected = (int)($kycRejected ?? 0); // optional
?>
<style>
  /* ---- Global safety rails (non-destructive) ---- */
  html, body { width: 100%; overflow-x: hidden; }         /* kill horiz scroll */
  * { box-sizing: border-box; }

  /* Your main content wrapper name may vary; keep one of these */
  .main-content, .content-inner, .wrap, .container-fluid { overflow-x: clip; }

  /* ---- Grid: 1 col by default, 2 cols only when there is room ---- */
  .dash-block{
    display: grid; gap: 16px;
    grid-template-columns: 1fr;              /* mobile/tablet: single column */
    min-width: 0;                             /* allow shrinking; no overflow */
  }
  /* Desktop: left expands, right column clamped (prevents overflow near sidebar) */
  @media (min-width: 1280px){
    .dash-block{
      grid-template-columns: minmax(0,1fr) clamp(320px, 32%, 420px);
    }
  }

  /* Ensure grid children can shrink */
  .dash-block > * { min-width: 0; }

  /* ---- Cards & rows inside must never force width ---- */
  .cardX, .stack { min-width: 0; }
  .hline { flex-wrap: wrap; gap: 8px; }
  .chip { white-space: nowrap; }              /* each chip wraps as a unit */

  /* ---- Chart responsiveness (prevents canvas oversizing) ---- */
  .chart-box{
    inline-size: 100%;
    height: 290px;               /* adjust per chart */
    overflow: hidden;            /* clip internal canvas overflow */
  }
  .chart-box > canvas{
    display: block;
    inline-size: 100% !important;
    block-size: 100% !important; /* height 100% of wrapper */
    max-width: 100% !important;
  }

  /* If you use multiple chart heights: tweak per instance */
  .chart-lg{ height: 300px; }
  .chart-md{ height: 260px; }

  /* Optional: slightly narrower split bar so borders/padding never push width */
  .splitbar{ width: 100%; }
</style>


<style>
  :root{
    --brand:#F1671B; --brand-700:#d85512;
    --ink:#0f172a; --muted:#64748b; --line:#f1f2f4;
    --green:#16a34a; --rose:#dc2626; --blue:#0ea5e9; --amber:#f59e0b;
    --soft:#fff6ef; --card:#fff; --shadow:0 12px 28px rgba(15,23,42,.06);
    --r-xl:18px; --r-lg:14px; --r-md:12px;
  }

  .dash-block{display:grid;gap:16px}
  /* 1 column mobile, 2 columns ≥ 1100px */
  @media (min-width:1100px){ .dash-block{ grid-template-columns: 2fr 1fr; } }

  .cardX{background:var(--card); border:1px solid var(--line); border-radius:var(--r-xl); box-shadow:var(--shadow); padding:16px}
  .cardX-h{display:flex;align-items:center;justify-content:space-between;gap:12px;margin-bottom:10px}
  .chip{display:inline-flex;align-items:center;gap:8px;padding:6px 10px;border-radius:999px;font-weight:700;font-size:12px}
  .chip-emerald{background:#ecfdf5;color:#065f46;border:1px solid #d1fae5}
  .chip-rose{background:#fef2f2;color:#7f1d1d;border:1px solid #fee2e2}
  .chip-blue{background:#eff6ff;color:#1e3a8a;border:1px solid #dbeafe}
  .chip-amber{background:#fffbeb;color:#92400e;border:1px solid #fef3c7}

  .hline{display:flex;gap:12px;flex-wrap:wrap}
  .title{display:flex;align-items:center;gap:10px;font-weight:800}
  .title i{color:var(--brand)}

  /* Split progress bar */
  .splitbar{position:relative;height:18px;border-radius:999px;background:#f9fafb;border:1px solid var(--line);overflow:hidden}
  .splitbar .dep{height:100%;background:linear-gradient(90deg,var(--brand),#ff914d)}
  .splitbar .wdr{position:absolute;right:0;top:0;height:100%;background:linear-gradient(90deg,#ffb74a,#ffdd9b)}

  /* Big stats (scrollable on small screens) */
  .mini-row{display:flex;gap:6px;overflow:auto;padding-bottom:4px;scroll-snap-type:x mandatory}
  .mini{min-width:280px;scroll-snap-align:start;border-radius:16px;padding:16px;border:1px solid var(--line)}
  .mini h4{margin:2px 0 0;font-size:16px;font-weight:600}
  .mini p{margin:4px 0 0;font-size:12px}

  .mini-emerald{background:#ecfdf5;border-color:#d1fae5}
  .mini-rose{background:#fef2f2;border-color:#fee2e2}
  .mini-blue{background:#eff6ff;border-color:#dbeafe}
  .mini-amber{background:#fffbeb;border-color:#fef3c7}

  .muted{color:var(--muted);font-size:12px}

  /* Right column: donut & monthly chart cards stack */
  .stack{display:grid;gap:16px} .mini-row{ display:flex; flex-wrap:wrap; gap:12px; justify-content:center; }
  .mini{ flex: 0 1 280px; max-width: 320px; }
  /* Hide bars on inner scrollers you use */
  .mini-row,
  .table-wrap,
  .chart-box {
    -ms-overflow-style: none;      /* IE/Edge */
    scrollbar-width: none;         /* Firefox */
  }
  .mini-row::-webkit-scrollbar,
  .table-wrap::-webkit-scrollbar,
  .chart-box::-webkit-scrollbar {
    display: none;                 /* Chrome/Safari/WebKit */
  }

  /* Optional: reduce overscroll bounce/glow */
  * { overscroll-behavior: contain; }

</style>

<section class="dash-block" style="margin-top:20px;">

  <!-- LEFT: Cash Flow Overview -->
  <div class="cardX">
    <div class="cardX-h">
      <div class="title"><i class="ri-exchange-dollar-fill"></i> <span>Cash Flow Overview</span></div>
      <div class="hline">
        <span class="chip chip-emerald"><i class="ri-arrow-up-circle-fill"></i> Deposits: $<?php echo number_format($dep,2); ?></span>
        <span class="chip chip-rose"><i class="ri-arrow-down-circle-fill"></i> Withdrawals: $<?php echo number_format($wdr,2); ?></span>
        <span class="chip <?php echo $net>=0?'chip-blue':'chip-amber'; ?>">
          <i class="ri-scales-3-fill"></i> Net: $<?php echo number_format($net,2); ?>
        </span>
      </div>
    </div>

    <!-- Split bar -->
    <div class="splitbar" title="Deposits <?php echo $depPct; ?>% · Withdrawals <?php echo $wdrPct; ?>%">
      <div class="dep" style="width: <?php echo $depPct; ?>%"></div>
      <div class="wdr" style="width: <?php echo $wdrPct; ?>%"></div>
    </div>
    <div class="muted" style="margin-top:6px">
      Deposits <?php echo $depPct; ?>% &nbsp;•&nbsp; Withdrawals <?php echo $wdrPct; ?>%
    </div>

    <!-- Monthly Trend -->
    <div class="cardX" style="margin-top:12px">
      <div class="title" style="margin-bottom:6px"><i class="ri-line-chart-fill"></i> Wallet Deposits vs Withdrawals (Monthly)</div>
      <div style="height:290px"><canvas id="walletTrendChart"></canvas></div>
    </div>

    <!-- Mini Stats row -->
    <div class="mini-row" style="margin-top:10px">
      <div class="mini mini-emerald">
        <div class="muted"><i class="ri-funds-fill"></i> Total Deposits</div>
        <h4>$<?php echo number_format($dep,2); ?></h4>
        <p class="muted">All-time cumulative deposits</p>
      </div>

      <div class="mini mini-rose">
        <div class="muted"><i class="ri-money-dollar-circle-fill"></i> Total Withdrawals</div>
        <h4>$<?php echo number_format($wdr,2); ?></h4>
        <p class="muted">All-time cumulative withdrawals</p>
      </div>

      <div class="mini <?php echo $net>=0?'mini-blue':'mini-amber'; ?>">
        <div class="muted"><i class="ri-bar-chart-2-fill"></i> Net Flow</div>
        <h4>$<?php echo number_format($net,2); ?></h4>
        <p class="muted">Deposits − Withdrawals</p>
      </div>
    </div>
  </div>

  <!-- RIGHT: KYC + Donut -->
  <div class="stack">
    <div class="cardX">
      <div class="title"><i class="ri-pass-valid-fill"></i> KYC Completion</div>
      <div style="height:300px"><canvas id="kycDonut"></canvas></div>
    </div>

    <div class="cardX">
      <div class="title"><i class="ri-bar-chart-grouped-fill"></i> Monthly Snapshot</div>
      <div style="height:260px"><canvas id="walletBars"></canvas></div>
    </div>
  </div>

</section>

<script>
  /* ---------- Monthly Deposits vs Withdrawals (line) ---------- */
  (function(){
    // If you have server arrays, echo them; otherwise we fallback.
    var months = <?php echo json_encode($walletMonths ?? []); ?>;
    var depS   = <?php echo json_encode($walletDepSeries ?? []); ?>;
    var wdrS   = <?php echo json_encode($walletWdrSeries ?? []); ?>;
    if(!months.length){
      months = ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'];
      // soft fallback series (proportional to totals just to render)
      var base = Math.max(1, <?php echo json_encode($dep + $wdr); ?>) / 12;
      depS = months.map((_,i)=> Math.round(base * (0.6 + Math.sin(i)*0.15 + 0.4)));
      wdrS = months.map((_,i)=> Math.round(base * (0.4 + Math.cos(i)*0.12 + 0.2)));
    }

    new Chart(document.getElementById('walletTrendChart'), {
      type: 'line',
      data: {
        labels: months,
        datasets: [
          {label: 'Deposits', data: depS, tension:.35, borderWidth:2, borderColor:'#F1671B', backgroundColor:'rgba(241,103,27,.10)', fill:true, pointRadius:0},
          {label: 'Withdrawals', data: wdrS, tension:.35, borderWidth:2, borderColor:'#ffb74a', backgroundColor:'rgba(255,183,74,.10)', fill:true, pointRadius:0}
        ]
      },
      options: {
        maintainAspectRatio:false,
        scales:{ y:{ grid:{ color:'#f3f4f6' } }, x:{ grid:{ display:false } } },
        plugins:{ legend:{ display:true } }
      }
    });
  })();

  /* ---------- KYC Donut ---------- */
  (function(){
    const kyc = {
      approved: <?php echo json_encode($kycApproved); ?>,
      pending : <?php echo json_encode($kycPending); ?>,
      rejected: <?php echo json_encode($kycRejected); ?>
    };
    new Chart(document.getElementById('kycDonut'), {
      type:'doughnut',
      data:{
        labels:['Approved','Pending','Rejected'],
        datasets:[{
          data:[kyc.approved, kyc.pending, kyc.rejected],
          backgroundColor:['#16a34a','#f59e0b','#ef4444'],
          borderColor:'#ffffff', borderWidth:2
        }]
      },
      options:{
        cutout:'62%',
        maintainAspectRatio:false,
        plugins:{
          legend:{ position:'bottom' }
        }
      }
    });
  })();

  /* ---------- Monthly Snapshot (bars) ---------- */
  (function(){
    var months = <?php echo json_encode($walletMonths ?? []); ?>;
    var depS   = <?php echo json_encode($walletDepSeries ?? []); ?>;
    var wdrS   = <?php echo json_encode($walletWdrSeries ?? []); ?>;
    if(!months.length){
      months = ['Jan','Feb','Mar','Apr','May','Jun'];
      depS = [120,150,180,160,200,220];
      wdrS = [80,90,110,100,130,140];
    }

    new Chart(document.getElementById('walletBars'), {
      type:'bar',
      data:{
        labels: months,
        datasets:[
          {label:'Deposits', data:depS, backgroundColor:'#F1671B', borderRadius:8, borderSkipped:false},
          {label:'Withdrawals', data:wdrS, backgroundColor:'#ffb74a', borderRadius:8, borderSkipped:false}
        ]
      },
      options:{
        maintainAspectRatio:false,
        scales:{ y:{ grid:{ color:'#f3f4f6' }, beginAtZero:true }, x:{ grid:{ display:false } } },
        plugins:{ legend:{ display:true } }
      }
    });
  })();
</script>
<style>
/* Mobile: turn table rows into labeled cards */
@media (max-width: 640px) {
  #topAccounts thead { display: none; }
  #topAccounts tbody tr {
    display: grid;
    grid-template-columns: 1fr;
    border-bottom: 1px solid #eee;
    padding: 6px 0;
  }
  #topAccounts tbody td {
    display: flex;
    justify-content: space-between;
    gap: 12px;
    padding: 8px 12px;
    white-space: nowrap;               /* keep short values on one line */
    overflow: hidden;
    text-overflow: ellipsis;
  }
  #topAccounts tbody td::before {
    content: attr(data-label);
    font-weight: 600;
    color: #64748b;
    margin-right: 10px;
    white-space: normal;
  }
}
</style>

      <!-- Tables Row -->
      <section class="mt-6 grid grid-cols-1 xl:grid-cols-2 gap-4">
        <div class="card rounded-2xl p-4 bg-white border border-gray-100 overflow-x-auto">
          <div class="flex items-center justify-between mb-2">
            <h3 class="text-base font-semibold mb-2 flex items-center gap-2">
  <i class="ri-trophy-fill text-amber-600"></i> Top 5 Live Accounts 
  <span class="text-xs px-2 py-0.5 rounded-lg bg-amber-50 text-amber-700 border border-amber-100">
    Live
  </span>
</h3>

          </div>
          <table id="topAccounts" class="display w-full text-sm">
            <thead>
              <tr>
                <th>Login ID</th>
                <th>Account</th>
                <th>Server</th>
                
              </tr>
            </thead>
            <tbody>
              <?php foreach ($topAccounts as $r): ?>
                <tr>
                  <td><?php echo htmlspecialchars($r['trade_id']); ?></td>
                  <td><?php echo htmlspecialchars($r['account_name']); ?></td>
                  <td><?php echo htmlspecialchars($r['server']); ?></td>
                  
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>

<!--        <div class="card rounded-2xl p-4 bg-white border border-gray-100 overflow-x-auto">-->
<!--          <div class="flex items-center justify-between mb-2">-->
<!--           <h3 class="text-base font-semibold mb-2 flex items-center gap-2">-->
<!--  <i class="ri-user-smile-fill text-brand-600"></i> Recent Users-->
<!--  <span class="text-xs px-2 py-0.5 rounded-lg bg-brand-50 text-brand-700 border border-brand-100">-->
<!--    Last signups-->
<!--  </span>-->
<!--</h3>-->

<!--          </div>-->
<!--          <table id="recentUsers" class="display w-full text-sm">-->
<!--            <thead>-->
<!--              <tr>-->
<!--                <th>Name</th>-->
<!--                <th>Email</th>-->
<!--                <th>Country</th>-->
<!--                <th>KYC</th>-->
<!--                <th>Client</th>-->
<!--                <th>Registered</th>-->
<!--              </tr>-->
<!--            </thead>-->
<!--            <tbody>-->
<!--              <?php foreach ($recentUsers as $u): ?>-->
<!--                <tr>-->
<!--                  <td><?php echo htmlspecialchars($u['fullname'] ?? '—'); ?></td>-->
<!--                  <td><?php echo htmlspecialchars($u['email'] ?? '—'); ?></td>-->
<!--                  <td><?php echo htmlspecialchars($u['country'] ?? '—'); ?></td>-->
<!--                  <td><?php echo ((int)($u['kyc_verify'] ?? 0)) ? '<span class="text-emerald-600 font-medium">Approved</span>' : '<span class="text-amber-600">Pending</span>'; ?></td>-->
<!--                  <td><?php echo ((int)($u['client_status'] ?? 0)) ? '<span class="text-emerald-600 font-medium">Active</span>' : '<span class="text-gray-500">Inactive</span>'; ?></td>-->
<!--                  <td><?php echo htmlspecialchars($u['reg_date']); ?></td>-->
<!--                </tr>-->
<!--              <?php endforeach; ?>-->
<!--            </tbody>-->
<!--          </table>-->
<!--        </div>-->
      </section>

     

      <?php include 'footer.php'; ?>
    </div>
  </main>
</div>

<script>
const totalDeposits = <?php echo json_encode((float)$totalDeposit); ?>;
const totalWithdraws = <?php echo json_encode((float)$totalWithdraw); ?>;
const kycApproved = <?=$kycApproved;?>;
const kycPending  = <?=$kycPending;?>;

// Blank chart if no data; otherwise 2 bars: Deposits vs Withdrawals
const hasTotals = (totalDeposits > 0 || totalWithdraws > 0);
new Chart(document.getElementById('depWdrChart'), {
  type: 'bar',
  data: hasTotals
    ? {
        labels: ['Deposits', 'Withdrawals'],
        datasets: [
          { label: 'Amount ($)', data: [totalDeposits, totalWithdraws] }
        ]
      }
    : { labels: [], datasets: [{ label: 'Amount ($)', data: [] }] },
  options: {
    responsive: true,
    maintainAspectRatio: false,
    scales: { y: { beginAtZero: true } },
    plugins: {
      legend: { display: false },
      tooltip: { callbacks: { label: (ctx) => '$' + Number(ctx.parsed.y).toLocaleString() } }
    }
  }
});


// DataTables init (kept)
$(function(){
  $('#topAccounts').DataTable({ pageLength: 5, lengthMenu:[5,10,25,50] });
  $('#recentUsers').DataTable({ pageLength: 8, lengthMenu:[8,16,32,64] });
});
</script>

<script>
const kycApproved = <?=$kycApproved;?>;
const kycPending  = <?=$kycPending;?>;
const kycNotSubmitted = <?=json_encode((int)$kycNotSubmitted)?>;

const hasAnyKyc = (kycApproved + kycPending + kycNotSubmitted) > 0;

new Chart(document.getElementById('kycPie'), {
  type: 'doughnut',
  data: {
    labels: ['Approved', 'Pending', 'Not Submitted'],
    datasets: [{ data: [kycApproved, kycPending, kycNotSubmitted] }]
  },
  options: {
    responsive: true,
    maintainAspectRatio: false,
    plugins: {
      legend: { position: 'bottom' },
      tooltip: {
        callbacks: {
          label: (ctx) => `${ctx.label}: ` + Number(ctx.parsed).toLocaleString()
        }
      }
    }
  }
});

// Optional message under the chart if absolutely no data (unlikely with Not Submitted)
if (!hasAnyKyc) {
  const p = document.createElement('p');
  p.className = 'text-center text-gray-400 text-sm mt-4';
  p.textContent = 'No KYC data available yet';
  document.getElementById('kycPie').parentElement.appendChild(p);
}
</script>

</body>
</html>