<?php
/**
 * manage_live_accounts.php
 * - Current leverage from liveaccount (by row)
 * - Dropdown options = all leverage where leverage.account_type = liveaccount.account_name
 * - Fully responsive (table for sm+; cards for mobile)
 * - Reliable POST via form attribute (no form-wrapping a <tr>)
 * - Uses header.php, side_bar.php, footer.php
 */
@session_start();

/* Debug (disable in prod) */
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/config.php'; // must set $conn = new PDO(...)
require_once __DIR__ . '/auth_guard.php'; // must set $conn = new PDO(...)

if (!isset($conn) || !($conn instanceof PDO)) {
  die('Database connection not established. Check config.php (PDO).');
}

/* ---------- Small PDO helpers ---------- */
function scalar(PDO $conn, string $sql, array $params = []) {
  $st = $conn->prepare($sql);
  if(!$st->execute($params)) return null;
  return $st->fetchColumn();
}
function rows(PDO $conn, string $sql, array $params = []): array {
  $st = $conn->prepare($sql);
  if(!$st->execute($params)) return [];
  return $st->fetchAll(PDO::FETCH_ASSOC);
}

/* -------------------------------------------
   Actions
-------------------------------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_live'])) {
  $tid   = (int)($_POST['trade_id'] ?? 0);
  $email = trim($_POST['email'] ?? '');          // kept for completeness (readonly in UI)
  $name  = trim($_POST['account_name'] ?? '');   // kept for completeness (readonly in UI)
  $lev   = trim($_POST['leverage'] ?? '');

  if ($tid > 0) {
    $sql = "UPDATE liveaccount 
            SET email = :email,
                account_name = :name,
                leverage = :lev
            WHERE trade_id = :tid";
    $stmt = $conn->prepare($sql);
    $stmt->execute([
      ':email' => $email !== '' ? $email : null,
      ':name'  => $name  !== '' ? $name  : null,
      ':lev'   => $lev   !== '' ? $lev   : null,
      ':tid'   => $tid
    ]);
  }
  header("Location: " . strtok($_SERVER['REQUEST_URI'], '?')); exit;
}

/* -------------------------------------------
   Data Fetch
-------------------------------------------- */
/* Live accounts we’ll display */
$liveRows = rows($conn, "
  SELECT trade_id, email, account_name, leverage
  FROM liveaccount
  ORDER BY trade_id DESC
");

/* Build list of needed account_name values (used as leverage.account_type) */
$namesNeeded = [];
foreach ($liveRows as $row) {
  $nameKey = trim((string)($row['account_name'] ?? ''));
  if ($nameKey !== '') $namesNeeded[$nameKey] = true;
}
$nameList = array_values(array_keys($namesNeeded));

/* Fetch leverage options grouped by account_type (== account_name) in ONE query */
$levByAccountName = [];
if (!empty($nameList)) {
  $placeholders = implode(',', array_fill(0, count($nameList), '?'));
  $sql = "SELECT TRIM(account_type) AS account_type, account_leverage
          FROM leverage
          WHERE TRIM(account_type) IN ($placeholders)
          ORDER BY account_type, account_leverage+0, account_leverage";
  $st = $conn->prepare($sql);
  $st->execute($nameList);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);
  foreach ($rows as $lr) {
    $t = $lr['account_type'];
    if ($t === '' || $t === null) continue;
    $levByAccountName[$t][] = $lr['account_leverage'];
  }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1.0"/>
  <title>Manage Live Accounts</title>

  <!-- Tailwind -->
  <script src="https://cdn.tailwindcss.com"></script>
  <script>
    tailwind.config = {
      theme: {
        extend: {
          colors: {
            brand: {
              50:'#eef7ff',100:'#d9edff',200:'#bfe0ff',300:'#94caff',400:'#63adff',
              500:'#3a90ff',600:'#2977e6',700:'#1f5fbe',800:'#1b4e98',900:'#183f79'
            }
          },
          boxShadow: { soft: '0 8px 24px rgba(0,0,0,.06)' }
        }
      }
    }
  </script>

  <!-- Icons (optional) -->
  <link href="https://cdnjs.cloudflare.com/ajax/libs/remixicon/4.2.0/remixicon.min.css" rel="stylesheet"/>

  <style>
    :root{ --vm-border:#e5e7eb; }
    .card{ background:#fff;border:1px solid var(--vm-border);border-radius:14px;box-shadow:0 8px 24px rgba(0,0,0,.06) }
    /* The form attribute pattern avoids wrapping <tr> with <form> */
  </style>
</head>
<body class="min-h-screen bg-white text-gray-900">
  <?php include __DIR__ . "/header.php"; ?>

  <div class="flex flex-col md:flex-row min-h-screen">
    <?php include __DIR__ . "/side_bar.php"; ?>

    <!-- Content -->
    <main class="flex-1 md:ml-64 px-4 sm:px-6 py-6 max-w-7xl mx-auto">
      <!-- Page title -->
      <div class="mb-4">
        <h1 class="text-2xl font-semibold tracking-tight">Manage Live Accounts</h1>
        
      </div>

      <!-- Manage Live Accounts -->
      <section class="mt-4">
        <div class="card p-4">
          <div class="flex items-center justify-between mb-3">
            <h3 class="font-semibold text-gray-800 text-lg">Live Accounts</h3>
            <span class="text-xs text-gray-500">Total: <?= count($liveRows) ?></span>
          </div>

          <!-- Desktop / Tablet: table view -->
          <div class="hidden sm:block overflow-x-auto">
            <table class="min-w-[880px] w-full text-sm">
              <thead>
                <tr class="bg-gray-50 text-gray-700">
                  <th class="text-left p-3 border-b">Trade ID</th>
                  <th class="text-left p-3 border-b">Email</th>
                  <th class="text-left p-3 border-b">Account Name</th>
                  <th class="text-left p-3 border-b">Leverage</th>
                  <th class="text-left p-3 border-b">Action</th>
                </tr>
              </thead>
              <tbody>
                <?php if (empty($liveRows)): ?>
                  <tr>
                    <td colspan="5" class="p-4 text-center text-gray-500">
                      No accounts in <code>liveaccount</code>.
                    </td>
                  </tr>
                <?php else: ?>
                  <?php foreach ($liveRows as $r): ?>
                    <?php
                      $fid      = 'f'.(int)$r['trade_id']; // unique form id per row
                      $acctName = trim((string)($r['account_name'] ?? ''));
                      $curLev   = trim((string)($r['leverage'] ?? ''));
                      $opts     = $levByAccountName[$acctName] ?? [];
                      if ($curLev !== '' && !in_array($curLev, $opts, true)) array_unshift($opts, $curLev);
                    ?>
                    <!-- Hidden real form (controls reference it via form="...") -->
                    <form id="<?= $fid ?>" method="POST" class="hidden">
                      <input type="hidden" name="trade_id" value="<?= (int)$r['trade_id'] ?>">
                    </form>

                    <tr class="odd:bg-white even:bg-gray-50 align-middle">
                      <td class="p-3 border-b font-medium text-gray-800">
                        <?= (int)$r['trade_id'] ?>
                      </td>

                      <td class="p-3 border-b">
                        <input type="email" name="email" form="<?= $fid ?>"
                          value="<?= htmlspecialchars($r['email'] ?? '') ?>"
                          class="w-full p-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-brand-200"
                          placeholder="user@example.com" readonly>
                      </td>

                      <td class="p-3 border-b">
                        <input type="text" name="account_name" form="<?= $fid ?>"
                          value="<?= htmlspecialchars($r['account_name'] ?? '') ?>"
                          class="w-full p-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-brand-200"
                          placeholder="Account name" readonly>
                      </td>

                      <td class="p-3 border-b">
                        <select name="leverage" form="<?= $fid ?>"
                          class="w-full p-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-brand-200">
                          <?php if (empty($opts)): ?>
                            <?php if ($curLev !== ''): ?>
                              <option value="<?= htmlspecialchars($curLev) ?>" selected><?= htmlspecialchars($curLev) ?></option>
                            <?php else: ?>
                              <option value="">— No options for this account —</option>
                            <?php endif; ?>
                          <?php else: foreach ($opts as $opt): ?>
                            <option value="<?= htmlspecialchars($opt) ?>" <?= ($opt === $curLev ? 'selected' : '') ?>>
                              <?= htmlspecialchars($opt) ?>
                            </option>
                          <?php endforeach; endif; ?>
                        </select>
                        <?php if ($acctName !== ''): ?>
                          <div class="mt-1 text-[11px] text-gray-500">options for: <?= htmlspecialchars($acctName) ?></div>
                        <?php endif; ?>
                      </td>

                      <td class="p-3 border-b">
                        <button type="submit" name="update_live" value="1" form="<?= $fid ?>"
                          class="px-3 py-2 rounded-md bg-brand-600 text-white hover:bg-brand-700">
                          Update
                        </button>
                      </td>
                    </tr>
                  <?php endforeach; ?>
                <?php endif; ?>
              </tbody>
            </table>
          </div>

          <!-- Mobile: card view -->
          <div class="space-y-3 sm:hidden">
            <?php if (empty($liveRows)): ?>
              <p class="p-4 text-center text-gray-500 bg-gray-50 rounded-lg">
                No accounts in <code>liveaccount</code>.
              </p>
            <?php else: ?>
              <?php foreach ($liveRows as $r): ?>
                <?php
                  $fid      = 'fm'.(int)$r['trade_id'];
                  $acctName = trim((string)($r['account_name'] ?? ''));
                  $curLev   = trim((string)($r['leverage'] ?? ''));
                  $opts     = $levByAccountName[$acctName] ?? [];
                  if ($curLev !== '' && !in_array($curLev, $opts, true)) array_unshift($opts, $curLev);
                ?>
                <form id="<?= $fid ?>" method="POST" class="hidden">
                  <input type="hidden" name="trade_id" value="<?= (int)$r['trade_id'] ?>">
                </form>

                <div class="border border-gray-200 rounded-lg p-3 bg-white">
                  <div class="flex items-center justify-between">
                    <div class="text-sm text-gray-500">Trade ID</div>
                    <div class="text-sm font-medium text-gray-900"><?= (int)$r['trade_id'] ?></div>
                  </div>

                  <label class="block mt-3 text-xs font-medium text-gray-600">Email</label>
                  <input type="email" name="email" form="<?= $fid ?>"
                    value="<?= htmlspecialchars($r['email'] ?? '') ?>"
                    class="w-full mt-1 p-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-brand-200"
                    placeholder="user@example.com" readonly>

                  <label class="block mt-3 text-xs font-medium text-gray-600">Account Name</label>
                  <input type="text" name="account_name" form="<?= $fid ?>"
                    value="<?= htmlspecialchars($r['account_name'] ?? '') ?>"
                    class="w-full mt-1 p-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-brand-200"
                    placeholder="Account name" readonly>

                  <label class="block mt-3 text-xs font-medium text-gray-600">Leverage</label>
                  <select name="leverage" form="<?= $fid ?>"
                    class="w-full mt-1 p-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-brand-200">
                    <?php if (empty($opts)): ?>
                      <?php if ($curLev !== ''): ?>
                        <option value="<?= htmlspecialchars($curLev) ?>" selected><?= htmlspecialchars($curLev) ?></option>
                      <?php else: ?>
                        <option value="">— No options for this account —</option>
                      <?php endif; ?>
                    <?php else: foreach ($opts as $opt): ?>
                      <option value="<?= htmlspecialchars($opt) ?>" <?= ($opt === $curLev ? 'selected' : '') ?>>
                        <?= htmlspecialchars($opt) ?>
                      </option>
                    <?php endforeach; endif; ?>
                  </select>

                  <?php if ($acctName !== ''): ?>
                    <div class="mt-1 text-[11px] text-gray-500">options for: <?= htmlspecialchars($acctName) ?></div>
                  <?php endif; ?>

                  <div class="mt-4">
                    <button type="submit" name="update_live" value="1" form="<?= $fid ?>"
                      class="w-full px-3 py-2 rounded-md bg-brand-600 text-white hover:bg-brand-700">
                      Update
                    </button>
                  </div>
                </div>
              <?php endforeach; ?>
            <?php endif; ?>
          </div>
        </div>
      </section>
    </main>
  </div>

  <?php include __DIR__ . "/footer.php"; ?>

  <script>
    /* Optional: user dropdown from your header, if needed */
    (function(){
      const btn = document.getElementById('userBtn');
      const menu = document.getElementById('userMenu');
      if(!btn || !menu) return;
      const toggle = ()=> menu.classList.toggle('hidden');
      btn.addEventListener('click', toggle);
      document.addEventListener('click', (e)=>{
        if(!menu.contains(e.target) && !btn.contains(e.target)){
          if(!menu.classList.contains('hidden')) menu.classList.add('hidden');
        }
      });
    })();
  </script>
</body>
</html>
