<?php
// create_ticket.php — Tailwind layout + your config + crmticket insert

if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }

// Include your PDO config ($conn). Tries config.php first, then "config (6).php".
$configIncluded = false;
foreach (['config.php', 'config (6).php'] as $cfg) {
  $path = __DIR__ . DIRECTORY_SEPARATOR . $cfg;
  if (file_exists($path)) { require_once $path; $configIncluded = true; break; }
}
if (!$configIncluded) { die("DB config file not found."); }
if (!isset($conn) || !($conn instanceof PDO)) { die("PDO \$conn missing from config."); }
$conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

// CSRF
if (empty($_SESSION['csrf_token'])) {
  $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Helpers
function clamp500($v){ $v = trim($v ?? ''); return mb_substr($v, 0, 500); }
function gen_ticketno(){
  $date = date('Ymd');
  $rand = strtoupper(substr(base_convert(bin2hex(random_bytes(4)), 16, 36), 0, 6));
  return "TKT-$date-$rand";
}
function ticketno_exists(PDO $conn, string $ticketno): bool {
  $q = $conn->prepare("SELECT 1 FROM crmticket WHERE ticketno = ? LIMIT 1");
  $q->execute([$ticketno]);
  return (bool)$q->fetchColumn();
}

// State
$errors = [];
$success = '';
$email = $subject = $type = $priority = $message = '';

// Handle POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if (!hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'] ?? '')) {
    $errors[] = "Invalid session. Please reload and try again.";
  }

  $email    = clamp500($_POST['email']    ?? '');
  $subject  = clamp500($_POST['subject']  ?? '');
  $type     = clamp500($_POST['type']     ?? '');
  $priority = clamp500($_POST['priority'] ?? '');
  $message  = clamp500($_POST['message']  ?? '');

  if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = "Valid email is required.";
  if ($subject === '')  $errors[] = "Subject is required.";
  if ($type === '')     $errors[] = "Type is required.";
  if ($priority === '') $errors[] = "Priority is required.";
  if ($message === '')  $errors[] = "Message is required.";

  if (!$errors) {
    // Generate unique ticketno
    $tries = 0; $exists = false; $ticketno = '';
    do {
      $ticketno = gen_ticketno();
      $exists = ticketno_exists($conn, $ticketno);
      $tries++;
    } while ($exists && $tries < 5);

    if ($exists) {
      $errors[] = "Could not generate a unique ticket number. Please submit again.";
    } else {
      try {
        $stmt = $conn->prepare("
          INSERT INTO crmticket
            (email, ticketno, subject, type, priority, message, status, AdminRemark)
          VALUES
            (:email, :ticketno, :subject, :type, :priority, :message, :status, :AdminRemark)
        ");
        $stmt->execute([
          ':email'       => $email,
          ':ticketno'    => $ticketno,
          ':subject'     => $subject,
          ':type'        => $type,
          ':priority'    => $priority,
          ':message'     => $message,
          ':status'      => 0,     // Open
          ':AdminRemark' => null   // Empty on create
        ]);
        $success = "Ticket created successfully! Your Ticket No is <strong>" . htmlspecialchars($ticketno) . "</strong>.";
        // Clear fields
        $email = $subject = $type = $priority = $message = '';
      } catch (Throwable $e) {
        $errors[] = "Insert failed: " . $e->getMessage();
      }
    }
  }
}

// dropdown options (free text allowed by schema; these are just UI helpers)
$types = ['Support','Bug','Feature','Billing','Other'];
$priorities = ['Low','Medium','High','Urgent'];
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8"/>
  <meta name="viewport" content="width=device-width,initial-scale=1"/>
  <title>Create Ticket</title>
  <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
</head>
<body class="bg-gray-50">
  <?php include 'header.php'; ?>

  <div class="flex flex-col md:flex-row min-h-screen">
    <?php if (file_exists(__DIR__ . '/side_bar.php')) include 'side_bar.php'; ?>

    <main class="flex-1 p-4 md:p-6 md:ml-64" style="width:100%;">
      <div class="max-w-5xl mx-auto">
        <div class="flex items-center justify-between mb-4">
          <h1 class="text-2xl font-semibold text-gray-800">Create Ticket</h1>
          <a href="my_ticket.php" class="text-sm text-blue-600 hover:underline">Back to Tickets</a>
        </div>

        <?php if ($success): ?>
          <div class="mb-4 rounded-md bg-green-50 p-4 border border-green-200">
            <div class="text-green-800 text-sm"><?= $success ?></div>
          </div>
        <?php endif; ?>

        <?php if ($errors): ?>
          <div class="mb-4 rounded-md bg-red-50 p-4 border border-red-200">
            <ul class="list-disc list-inside text-red-800 text-sm">
              <?php foreach ($errors as $err): ?>
                <li><?= htmlspecialchars($err, ENT_QUOTES, 'UTF-8'); ?></li>
              <?php endforeach; ?>
            </ul>
          </div>
        <?php endif; ?>

        <div class="bg-white p-6 md:p-8 rounded-lg shadow">
          <form method="post" class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($_SESSION['csrf_token']) ?>"/>

            <div class="md:col-span-2">
              <label class="block text-sm font-medium text-gray-700 mb-1">Email <span class="text-red-500">*</span></label>
              <input type="email" name="email" required
                     value="<?= htmlspecialchars($email, ENT_QUOTES, 'UTF-8'); ?>"
                     class="w-full p-3 border border-gray-300 rounded-md focus:border-blue-500 focus:ring-blue-500 outline-none">
            </div>

            <div class="md:col-span-2">
              <label class="block text-sm font-medium text-gray-700 mb-1">Subject <span class="text-red-500">*</span></label>
              <input type="text" name="subject" required
                     value="<?= htmlspecialchars($subject, ENT_QUOTES, 'UTF-8'); ?>"
                     class="w-full p-3 border border-gray-300 rounded-md focus:border-blue-500 focus:ring-blue-500 outline-none">
              <p class="text-xs text-gray-500 mt-1">Max 500 characters (per DB schema).</p>
            </div>

            <div>
              <label class="block text-sm font-medium text-gray-700 mb-1">Type <span class="text-red-500">*</span></label>
              <select name="type" required
                      class="w-full p-3 border border-gray-300 rounded-md focus:border-blue-500 focus:ring-blue-500 outline-none">
                <option value="">Select…</option>
                <?php foreach ($types as $opt): ?>
                  <option value="<?= htmlspecialchars($opt) ?>" <?= $type===$opt?'selected':''; ?>><?= htmlspecialchars($opt) ?></option>
                <?php endforeach; ?>
              </select>
            </div>

            <div>
              <label class="block text-sm font-medium text-gray-700 mb-1">Priority <span class="text-red-500">*</span></label>
              <select name="priority" required
                      class="w-full p-3 border border-gray-300 rounded-md focus:border-blue-500 focus:ring-blue-500 outline-none">
                <option value="">Select…</option>
                <?php foreach ($priorities as $opt): ?>
                  <option value="<?= htmlspecialchars($opt) ?>" <?= $priority===$opt?'selected':''; ?>><?= htmlspecialchars($opt) ?></option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="md:col-span-2">
              <label class="block text-sm font-medium text-gray-700 mb-1">Message <span class="text-red-500">*</span></label>
              <textarea name="message" rows="8" required
                        class="w-full p-3 border border-gray-300 rounded-md focus:border-blue-500 focus:ring-blue-500 outline-none"><?= htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?></textarea>
              <p class="text-xs text-gray-500 mt-1">Max 500 characters (per DB schema).</p>
            </div>

            <div class="md:col-span-2 flex items-center justify-end gap-3">
              <a href="my_ticket.php" class="px-4 py-2 bg-gray-100 text-gray-800 rounded-md hover:bg-gray-200">Cancel</a>
              <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">Create Ticket</button>
            </div>
          </form>
        </div>

        <p class="text-xs text-gray-500 mt-4">On create: <code>status=0 (Open)</code>, <code>AdminRemark=NULL</code>. Timestamp saved by column default.</p>
      </div>
    </main>
  </div>
</body>
</html>
