<?php
// my_ticket.php — Tickets list for crmticket schema
if (session_status() === PHP_SESSION_NONE) { session_start(); }
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/config.php';

/* Normalize PDO handle */
if (!isset($conn) || !($conn instanceof PDO)) {
  if (isset($dbh) && $dbh instanceof PDO) { $conn = $dbh; }
  else { die('No PDO connection found in config.php'); }
}

/* Require login */
if (empty($_SESSION['clogin'])) { header('Location: login.php'); exit; }
$userEmail = $_SESSION['clogin'];

/* Helpers */
function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
function qs(array $over = []) { $q = array_merge($_GET, $over); return '?' . http_build_query($q); }

/* ---------- Inputs ---------- */
$status   = isset($_GET['status']) ? (int)$_GET['status'] : 0; // 0/other = All
$search   = isset($_GET['search']) ? trim($_GET['search']) : '';
$per_page = isset($_GET['per_page']) ? max(1,(int)$_GET['per_page']) : 10;
$page     = isset($_GET['page']) ? max(1,(int)$_GET['page']) : 1;
$start    = ($page - 1) * $per_page;

/* ---------- WHERE (adaptive by status) ----------
   Table: crmticket (alias t)
   Columns: id, email, ticketno, subject, type, priority, message, status, AdminRemark, Date
*/
$where = "WHERE t.email = :email
  AND (
        (:status = 1 AND t.status = 1)
     OR (:status = 2 AND t.status = 2)
     OR (:status NOT IN (1,2))
  )";

$params = [
  ':email'  => $userEmail,
  ':status' => $status,
];

/* Optional search across ticketno / subject / type / priority */
if ($search !== '') {
  $where .= " AND (t.ticketno LIKE :q OR t.subject LIKE :q OR t.type LIKE :q OR t.priority LIKE :q)";
  $params[':q'] = "%{$search}%";
}

/* ---------- Count for pagination ---------- */
$sql_count = "SELECT COUNT(*) FROM crmticket t $where";
$stc = $conn->prepare($sql_count);
foreach ($params as $k=>$v) {
  if ($k === ':status') $stc->bindValue($k, (int)$v, PDO::PARAM_INT);
  else $stc->bindValue($k, $v);
}
$stc->execute();
$total_rows  = (int)$stc->fetchColumn();
$total_pages = (int)ceil($total_rows / $per_page);

/* ---------- Fetch page ---------- */
$sql = "SELECT
          t.id,
          t.ticketno,
          t.subject,
          t.type,
          t.priority,
          t.status,
          t.AdminRemark,
          t.`Date` AS ticket_date
        FROM crmticket t
        $where
        ORDER BY t.`Date` DESC, t.id DESC
        LIMIT :start, :per_page";

$st = $conn->prepare($sql);
foreach ($params as $k=>$v) {
  if ($k === ':status') $st->bindValue($k, (int)$v, PDO::PARAM_INT);
  else $st->bindValue($k, $v);
}
$st->bindValue(':start', (int)$start, PDO::PARAM_INT);     // MySQL LIMIT must be INT
$st->bindValue(':per_page', (int)$per_page, PDO::PARAM_INT);
$st->execute();
$tickets = $st->fetchAll(PDO::FETCH_ASSOC);

/* Badges */
function status_badge($s) {
  $s = (int)$s;
  if ($s === 1) return '<span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-semibold bg-emerald-100 text-emerald-700">Open</span>';
  if ($s === 2) return '<span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-semibold bg-slate-200 text-slate-700">Closed</span>';
  return '<span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-semibold bg-amber-100 text-amber-800">Other</span>';
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8"/>
  <meta name="viewport" content="width=device-width, initial-scale=1"/>
  <title>My Tickets</title>
  <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/remixicon@3.5.0/fonts/remixicon.css" rel="stylesheet">
</head>
<body class="bg-gray-50">
  <?php include 'header.php'; ?>
  <?php include 'side_bar.php'; ?>

  <main class="flex-1 p-4 md:p-6 md:ml-64">
    <div class="flex items-center justify-between mb-4">
      <h1 class="text-2xl md:text-3xl font-bold text-[#CD7F32]">My Tickets</h1>
      <form method="get" class="flex items-center gap-2">
        <input type="hidden" name="status" value="<?= (int)$status ?>">
        <input type="text" name="search" value="<?= h($search) ?>" placeholder="Search: ticket no / subject / type / priority"
               class="px-3 py-2 border border-gray-300 rounded text-sm bg-white">
        <select name="per_page" class="px-2 py-2 border border-gray-300 rounded text-sm bg-white" onchange="this.form.submit()">
          <option value="10" <?= $per_page==10?'selected':''; ?>>10</option>
          <option value="25" <?= $per_page==25?'selected':''; ?>>25</option>
          <option value="50" <?= $per_page==50?'selected':''; ?>>50</option>
        </select>
        <button class="px-3 py-2 bg-[#CD7F32] text-white rounded text-sm">Apply</button>
        <?php if ($search !== ''): ?>
          <a href="<?= h(qs(['search'=>null,'page'=>1])) ?>" class="px-3 py-2 bg-gray-200 rounded text-sm">Clear</a>
        <?php endif; ?>
      </form>
    </div>

    <!-- Tabs -->
    <div class="flex flex-wrap gap-2 mb-3">
      <a href="<?= h(qs(['status'=>0,'page'=>1])) ?>"
         class="px-3 py-1.5 rounded-lg text-sm font-semibold <?= $status===0?'bg-amber-600 text-white':'bg-gray-100 text-gray-800' ?>">
        <i class="ri-list-check-2"></i> All
      </a>
      <a href="<?= h(qs(['status'=>1,'page'=>1])) ?>"
         class="px-3 py-1.5 rounded-lg text-sm font-semibold <?= $status===1?'bg-amber-600 text-white':'bg-gray-100 text-gray-800' ?>">
        <i class="ri-checkbox-circle-line"></i> Open
      </a>
      <a href="<?= h(qs(['status'=>2,'page'=>1])) ?>"
         class="px-3 py-1.5 rounded-lg text-sm font-semibold <?= $status===2?'bg-amber-600 text-white':'bg-gray-100 text-gray-800' ?>">
        <i class="ri-lock-line"></i> Closed
      </a>
    </div>

    <!-- Table -->
    <div class="bg-white border border-gray-200 rounded-lg shadow overflow-x-auto">
      <table class="w-full text-sm min-w-[820px]">
        <thead>
          <tr class="bg-gray-100 text-left text-[#CD7F32]">
            <th class="px-3 py-2 border-b">Ticket No</th>
            <th class="px-3 py-2 border-b">Subject</th>
            <th class="px-3 py-2 border-b">Type</th>
            <th class="px-3 py-2 border-b">Priority</th>
            <th class="px-3 py-2 border-b">Status</th>
            <th class="px-3 py-2 border-b">Date</th>
            <th class="px-3 py-2 border-b">Action</th>
          </tr>
        </thead>
        <tbody>
          <?php if ($tickets): foreach ($tickets as $t): ?>
            <tr class="hover:bg-gray-50">
              <td class="px-3 py-2 border-b"><?= h($t['ticketno']) ?></td>
              <td class="px-3 py-2 border-b"><?= h($t['subject']) ?></td>
              <td class="px-3 py-2 border-b"><?= h($t['type']) ?></td>
              <td class="px-3 py-2 border-b"><?= h($t['priority']) ?></td>
              <td class="px-3 py-2 border-b"><?= status_badge($t['status']) ?></td>
              <td class="px-3 py-2 border-b"><?= h($t['ticket_date']) ?></td>
              <td class="px-3 py-2 border-b">
                <a class="px-3 py-1 rounded bg-[#CD7F32] text-white text-xs" href="ticket_view.php?id=<?= (int)$t['id'] ?>">
                  View
                </a>
              </td>
            </tr>
          <?php endforeach; else: ?>
            <tr><td colspan="7" class="px-3 py-6 text-center text-gray-600">No tickets found.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <!-- Pagination -->
    <div class="mt-4 flex items-center justify-between text-xs text-gray-600">
      <span>Showing <?= $total_rows ? ($start+1) : 0 ?>–<?= min($start+$per_page, $total_rows) ?> of <?= $total_rows ?></span>
      <div class="flex gap-2">
        <?php if ($page > 1): ?>
          <a class="px-3 py-1 bg-gray-100 rounded" href="<?= h(qs(['page'=>$page-1])) ?>">Prev</a>
        <?php endif; ?>
        <?php if ($page < $total_pages): ?>
          <a class="px-3 py-1 bg-gray-100 rounded" href="<?= h(qs(['page'=>$page+1])) ?>">Next</a>
        <?php endif; ?>
      </div>
    </div>

    <br><br><br>
    <?php include 'footer.php'; ?>
  </main>
</body>
</html>
