<?php
@session_start();
require_once 'config.php'; // must define $conn = new PDO(...)
require_once 'auth_guard.php'; 
// ---- Require login ----


// fetch helper
function fetch_admin(PDO $db, string $identity, bool $byEmail): ?array {
  if ($byEmail) {
    $sql = "SELECT * FROM emplist WHERE email = :v LIMIT 1";
  } else {
    $sql = "SELECT * FROM emplist WHERE empId = :v LIMIT 1";
  }
  $st = $db->prepare($sql);
  $st->execute([':v' => $identity]);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  return $row ?: null;
}

// ---- Load admin row ----
$admin = fetch_admin($conn, $loginIdentity, $isEmailLogin);
if (!$admin) {
  // As a fallback, if clogin looked like email but didn't match, try as empId (or vice versa)
  $admin = fetch_admin($conn, $loginIdentity, !$isEmailLogin);
  if ($admin) { $isEmailLogin = !$isEmailLogin; } // correct the flag
}
if (!$admin) { die('No admin found for: '.htmlspecialchars($loginIdentity)); }

// ---- Flash holder ----
$flash = ['type'=>null,'msg'=>null];

// ---- Handle POST update ----
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  // CSRF
  if (!hash_equals($_SESSION['csrf_profile'] ?? '', $_POST['csrf'] ?? '')) {
    $flash = ['type'=>'error','msg'=>'Invalid session token. Please reload and try again.'];
  } else {
    // Inputs
    $newName  = trim($_POST['name']  ?? '');
    $newEmail = trim($_POST['email'] ?? '');
    // NOTE: schema has varchar(20) for password; keeping plaintext per your schema
    $newPass  = $_POST['password'] ?? '';

    // Basic validation
    if ($newName === '' || $newEmail === '') {
      $flash = ['type'=>'error','msg'=>'Name and Email are required.'];
    } elseif (!filter_var($newEmail, FILTER_VALIDATE_EMAIL)) {
      $flash = ['type'=>'error','msg'=>'Invalid email format.'];
    } else {
      // Unique email check (except current row)
      $chk = $conn->prepare("SELECT id FROM emplist WHERE email = :e AND id <> :id LIMIT 1");
      $chk->execute([':e'=>$newEmail, ':id'=>$admin['id']]);
      if ($chk->fetch()) {
        $flash = ['type'=>'error','msg'=>'That email is already in use by another user.'];
      }
    }

    // File upload (optional)
    $profilePath = $admin['profile_pic']; // keep old by default
    if ($flash['type'] === null && !empty($_FILES['profile_picture']['name'])) {
      $f = $_FILES['profile_picture'];
      if ($f['error'] === UPLOAD_ERR_OK) {
        if ($f['size'] > 3 * 1024 * 1024) {
          $flash = ['type'=>'error','msg'=>'Profile image must be ≤ 3 MB.'];
        } else {
          $fi = new finfo(FILEINFO_MIME_TYPE);
          $mime = $fi->file($f['tmp_name']);
          $allowed = ['image/jpeg'=>'jpg','image/png'=>'png','image/webp'=>'webp'];
          if (!isset($allowed[$mime])) {
            $flash = ['type'=>'error','msg'=>'Only JPG, PNG or WebP images are allowed.'];
          } else {
            $dirAbs = __DIR__ . '/uploads/profile_pics';
            if (!is_dir($dirAbs)) { @mkdir($dirAbs, 0755, true); }
            $ext = $allowed[$mime];
            $fname = 'admin_'.$admin['id'].'_'.time().'.'.$ext;
            $destAbs = $dirAbs . '/' . $fname;
            $destRel = 'uploads/profile_pics/' . $fname;
            if (move_uploaded_file($f['tmp_name'], $destAbs)) {
              $profilePath = $destRel;
            } else {
              $flash = ['type'=>'error','msg'=>'Failed to save the uploaded file.'];
            }
          }
        }
      } else {
        $flash = ['type'=>'error','msg'=>'Upload error (code '.$f['error'].').'];
      }
    }

    // Proceed with UPDATE if no errors so far
    if ($flash['type'] === null) {
      // keep old password if empty
      $finalPass = ($newPass === '') ? $admin['password'] : $newPass;

      // Build and run update
      $params = [
        ':u'  => $newName,
        ':e'  => $newEmail,
        ':p'  => $finalPass,
        ':id' => $admin['id'],
      ];

      if ($profilePath !== $admin['profile_pic']) {
        $sqlUpd = "UPDATE emplist
                   SET username = :u, email = :e, password = :p, profile_pic = :pic
                   WHERE id = :id LIMIT 1";
        $params[':pic'] = $profilePath;
      } else {
        $sqlUpd = "UPDATE emplist
                   SET username = :u, email = :e, password = :p
                   WHERE id = :id LIMIT 1";
      }

      $upd = $conn->prepare($sqlUpd);
      $ok  = $upd->execute($params);

      if ($ok) {
        // If login identity is email, refresh sessions when email changes
        if ($isEmailLogin && $newEmail !== $loginIdentity) {
          $_SESSION['clogin'] = $newEmail;
        }
        // Always keep cemail in sync if you use it elsewhere
        $_SESSION['cemail'] = $newEmail;

        // Reload row
        $admin = fetch_admin($conn, $isEmailLogin ? $_SESSION['clogin'] : $loginIdentity, $isEmailLogin);
        if (!$admin) {
          // If not found under email (e.g., you switched), try empId fallback
          $admin = fetch_admin($conn, $loginIdentity, !$isEmailLogin);
        }

        $flash = ['type'=>'success','msg'=>'Profile updated successfully.'];
      } else {
        $flash = ['type'=>'error','msg'=>'Update failed. Please try again.'];
      }
    }
  }
}

// Helper for safe output
function e($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }

?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Admin Profile</title>
  <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100 flex flex-col items-center p-6">

  <div class="w-full max-w-4xl bg-white p-6 rounded-lg shadow-md">
    <h1 class="text-2xl font-semibold text-orange-600 mb-6">Admin Profile</h1>

    <?php if ($flash['type']): ?>
      <div class="mb-4 rounded-md p-3 text-sm <?= $flash['type']==='success' ? 'bg-green-50 text-green-700 border border-green-200' : 'bg-rose-50 text-rose-700 border border-rose-200' ?>">
        <?= e($flash['msg']) ?>
      </div>
    <?php endif; ?>

    <form action="profile.php" method="POST" enctype="multipart/form-data">
      <input type="hidden" name="csrf" value="<?= e($csrf) ?>">

      <!-- Admin Profile Image -->
      <div class="flex justify-center mb-4">
        <?php
          $img = $admin['profile_pic'];
          if (!$img || !is_string($img) || $img === '') $img = 'default.jpg';
        ?>
        <img src="<?= e($img) ?>" alt="Admin Profile"
             class="w-32 h-32 rounded-full border-2 border-orange-600 object-cover">
      </div>

      <!-- Profile Picture Upload -->
      <div class="mb-4">
        <label for="profile_picture" class="block text-sm font-medium text-gray-700">Change Profile Picture</label>
        <input type="file" id="profile_picture" name="profile_picture"
               accept="image/jpeg,image/png,image/webp"
               class="mt-1 p-2 border border-gray-300 rounded-md w-full">
        <p class="text-xs text-gray-500 mt-1">Allowed: JPG, PNG, WebP • Max 3 MB</p>
      </div>

      <!-- Name (emplist.username) -->
      <div class="mb-4">
        <label for="name" class="block text-sm font-medium text-gray-700">Name</label>
        <input type="text" id="name" name="name"
               value="<?= e($admin['username']) ?>"
               class="mt-1 p-2 border border-gray-300 rounded-md w-full" required>
      </div>

      <!-- Email (emplist.email) -->
      <div class="mb-4">
        <label for="email" class="block text-sm font-medium text-gray-700">Email</label>
        <input type="email" id="email" name="email"
               value="<?= e($admin['email']) ?>"
               class="mt-1 p-2 border border-gray-300 rounded-md w-full" required>
      </div>

      <!-- Password (emplist.password) -->
      <div class="mb-4">
        <label for="password" class="block text-sm font-medium text-gray-700">Password</label>
        <input type="text" id="password" name="password"
               value=""
               class="mt-1 p-2 border border-gray-300 rounded-md w-full"
               placeholder="Leave blank to keep current">
        <p class="text-xs text-gray-500 mt-1">Leave blank to keep the existing password.</p>
      </div>

      <div class="flex justify-between items-center mt-6">
        <button type="submit" class="bg-orange-600 text-white py-2 px-4 rounded-md hover:opacity-90">
          Update Profile
        </button>
      </div>
    </form>
  </div>

</body>
</html>
