<?php
session_start();
error_reporting(0);
require_once 'config.php'; // must define $conn (PDO)

if (empty($_SESSION['role']) || $_SESSION['role'] !== 'admin') { header('Location: login.php'); exit; }

header('X-Content-Type-Options: nosniff');

// -------- Handle AJAX: Save Bank Details (with optional file upload) --------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax']) && $_POST['ajax'] === 'save_bank') {
    header('Content-Type: application/json; charset=utf-8');

    try {
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) {
            echo json_encode(['ok'=>false,'error'=>'Invalid user id']); exit;
        }

        $bank_name            = trim((string)($_POST['bank_name'] ?? ''));
        $account_holder_name  = trim((string)($_POST['account_holder_name'] ?? ''));
        $bank_account_no      = trim((string)($_POST['bank_account_no'] ?? ''));
        $IFSC_Code            = trim((string)($_POST['IFSC_Code'] ?? ''));
        $swift_code           = trim((string)($_POST['swift_code'] ?? ''));

        // Build dynamic SET parts
        $set = [
            "bank_name = :bank_name",
            "account_holder_name = :account_holder_name",
            "bank_account_no = :bank_account_no",
            "IFSC_Code = :IFSC_Code",
            "swift_code = :swift_code",
            "bank_status = 1"
        ];
        $params = [
            ':id' => $id,
            ':bank_name' => $bank_name,
            ':account_holder_name' => $account_holder_name,
            ':bank_account_no' => $bank_account_no,
            ':IFSC_Code' => $IFSC_Code,
            ':swift_code' => $swift_code
        ];

        // Optional file upload (bank_detail)
        $uploadedPath = null;
        if (!empty($_FILES['bank_detail']['name']) && is_uploaded_file($_FILES['bank_detail']['tmp_name'])) {
            $uploadDir = __DIR__ . '/_docs/';
            if (!is_dir($uploadDir)) {
                @mkdir($uploadDir, 0755, true);
            }
            $orig = $_FILES['bank_detail']['name'];
            $ext  = strtolower(pathinfo($orig, PATHINFO_EXTENSION));

            $allowed = ['jpg','jpeg','png','webp','gif','pdf'];
            if (!in_array($ext, $allowed, true)) {
                echo json_encode(['ok'=>false,'error'=>'Only jpg, jpeg, png, webp, gif, pdf allowed']); exit;
            }

            $newName = 'bank_'.$id.'_'.time().'.'.$ext;
            $destAbs = $uploadDir . $newName;
            if (!move_uploaded_file($_FILES['bank_detail']['tmp_name'], $destAbs)) {
                echo json_encode(['ok'=>false,'error'=>'Failed to move uploaded file']); exit;
            }
            $uploadedPath = '_docs/'.$newName; // relative path stored in DB
            $set[] = "bank_detail = :bank_detail";
            $params[':bank_detail'] = $uploadedPath;
        }

        $sql = "UPDATE aspnetusers SET ".implode(', ', $set)." WHERE id = :id";
        $stmt = $conn->prepare($sql);
        foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
        $stmt->execute();

        echo json_encode([
            'ok' => true,
            'row' => [
                'id' => $id,
                'bank_name' => $bank_name,
                'account_holder_name' => $account_holder_name,
                'bank_account_no' => $bank_account_no,
                'IFSC_Code' => $IFSC_Code,
                'swift_code' => $swift_code,
                'bank_detail' => $uploadedPath
            ]
        ]);
    } catch (Throwable $e) {
        echo json_encode(['ok'=>false,'error'=>$e->getMessage()]);
    }
    exit;
}

// -------- Fetch users --------
function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }

$users = [];
try {
$q = $conn->query("SELECT id, email, fullname, bank_name, account_holder_name, bank_account_no, IFSC_Code, swift_code, bank_detail, bank_status FROM aspnetusers ORDER BY id DESC");
    $users = $q->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
    $errFetch = $e->getMessage();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Bank Details List</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        .container { margin-bottom: 1.5rem; display: flex; flex-direction: column; gap: 1rem; }
        .table { width: 100%; border-collapse: collapse; background: #fff; box-shadow: 0 4px 6px rgba(0,0,0,.1); border-radius: 8px; overflow: hidden; }
        .table thead { background: #CD7F32; color: green; }
        .table th { padding: 12px 16px; text-align: left; font-weight: 600; text-transform: uppercase; font-size: 14px; letter-spacing: .5px; }
        .table td { padding: 12px 16px; border-bottom: 1px solid #e5e7eb; vertical-align: middle; }
        .table tbody tr:hover { background: #f9fafb; }

        .badge { display: inline-block; padding: 5px 10px; font-size: 12px; border-radius: 20px; font-weight: 600; }
        .badge.green { background:#10B981; color:#fff; }
        .badge.red { background:#EF4444; color:#fff; }

        .modal { display:none; position: fixed; inset:0; background: rgba(0,0,0,.5); z-index: 1000; align-items: center; justify-content: center; }
        .modal-inner { background: #fff; border-radius: 12px; width: 95%; max-width: 700px; max-height: 90vh; overflow-y: auto; box-shadow: 0 20px 25px -5px rgba(0,0,0,.1); }
        .modal-header { padding: 16px 20px; border-bottom: 1px solid #e5e7eb; display: flex; justify-content: space-between; align-items: center; }
        .modal-title { font-size: 18px; font-weight: 700; color: #DC2626; }
        .modal-body { padding: 20px; background: #f3f4f6; }
        .modal-footer { padding: 16px 20px; border-top: 1px solid #e5e7eb; text-align: right; background:#fff; }
        .close-btn { font-size: 24px; line-height: 1; color:#6b7280; cursor:pointer; border:none; background:transparent; }
        .close-btn:hover { color:#111827; }

        @media (min-width: 768px) {
            .container { flex-direction: row; justify-content: space-between; align-items: center; }
            #search { width: 33.333%; }
            #status-filter { width: 25%; }
        }
        @media (max-width: 767px) {
            .table { display: block; overflow-x: auto; }
            .table th, .table td { padding: 8px 12px; font-size: 14px; }
        }
    </style>
</head>
<body class="bg-gray-100 text-sm md:text-base">
    <?php include "header.php";?>
    <div class="flex flex-col md:flex-row h-screen">
        <?php include "side_bar.php";?>
        <main class="flex-1 p-4 md:p-6 md:ml-64">
        <div class="container">
            <input id="search" type="text" placeholder="Search..." class="p-2 border border-gray-300 rounded-lg w-full md:w-1/3 mb-4 md:mb-0 focus:ring-2 focus:ring-[#CD7F32]">
            <select id="status-filter" class="p-2 border border-gray-300 rounded-lg w-full md:w-1/4 focus:ring-2 focus:ring-[#CD7F32]">
                <option value="">All Status</option>
                <option value="1">Updated</option>
                <option value="0">Not Updated</option>
            </select>
        </div>
 <h2 class="text-2xl font-bold text-[#CD7F32] mb-4">Bank Details List</h2>
        <div class="container mx-auto p-0 md:p-6">
           

            <div class="overflow-x-auto">
                <table id="bankTable" class="table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Email</th>
                            <th>Name</th>
                            <th>Bank</th>
                            <th>Account Holder</th>
                            <th>Acc No</th>
                            <th>IFSC</th>
                            <th>SWIFT</th>
                            <th>Status</th>
                            <th>Proof</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($users as $u):
                        $status = (int)$u['bank_status'];
                        $statusBadge = $status === 1
                            ? '<span class="badge green">Updated</span>'
                            : '<span class="badge red">Not Updated</span>';
                        $proof = $u['bank_detail'] ? h($u['bank_detail']) : '';
                    ?>
                        <tr data-bank-status="<?= $status ?>">
                            <td><?= (int)$u['id'] ?></td>
                            <td><?= h($u['email']) ?></td>
                            <td><?= h($u['fullname']) ?></td>
                            <td class="td-bank_name"><?= h($u['bank_name']) ?></td>
                            <td class="td-account_holder_name"><?= h($u['account_holder_name']) ?></td>
                            <td class="td-bank_account_no"><?= h($u['bank_account_no']) ?></td>
                            <td class="td-IFSC_Code"><?= h($u['IFSC_Code']) ?></td>
                            <td class="td-swift_code"><?= h($u['swift_code']) ?></td>
                            <td class="td-status"><?= $statusBadge ?></td>
                            <td class="td-proof">
                                <?php if ($proof): ?>
                                   <button type="button"
  class="px-3 py-1 rounded bg-blue-600 text-white text-sm hover:bg-blue-700 view-proof"
  data-src="<?= h($proof) ?>"
  onclick="openProofModal(this)"
>
  View
</button>

                                <?php else: ?>
                                    <span class="text-gray-400 text-sm">—</span>
                                <?php endif; ?>
                            </td>
                            <td>
  <button type="button"
    class="px-3 py-1 rounded bg-amber-600 text-white text-sm hover:bg-amber-700 edit-bank"
    data-id="<?= (int)$u['id'] ?>"
    data-email="<?= h($u['email']) ?>"
    data-fullname="<?= h($u['fullname']) ?>"
    data-bank_name="<?= h($u['bank_name']) ?>"
    data-account_holder_name="<?= h($u['account_holder_name']) ?>"
    data-bank_account_no="<?= h($u['bank_account_no']) ?>"
    data-IFSC_Code="<?= h($u['IFSC_Code']) ?>"
    data-swift_code="<?= h($u['swift_code']) ?>"
    data-proof="<?= h($proof) ?>"
    onclick="openBankModal(this)"
  >
    <?= $status ? 'Update' : 'Add' ?>
  </button>
</td>

                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <?php if (!empty($errFetch)): ?>
                <div class="mt-4 p-3 rounded bg-red-100 text-red-700">Error: <?= h($errFetch) ?></div>
            <?php endif; ?>
        </div>
    </main>
</div>

<!-- Edit/Add Bank Modal -->
<div id="bankModal" class="modal">
  <div class="modal-inner">
    <div class="modal-header">
      <div class="modal-title">Bank Details</div>
      <button class="close-btn" id="closeBankModal">&times;</button>
    </div>
    <form id="bankForm" enctype="multipart/form-data">
      <input type="hidden" name="ajax" value="save_bank">
      <input type="hidden" name="id" id="f_id">
      <div class="modal-body">
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <label class="block text-sm text-gray-700">User</label>
            <input id="f_user_label" type="text" class="w-full border rounded p-2 bg-gray-100" readonly>
          </div>
          <div>
            <label class="block text-sm text-gray-700">Bank Name</label>
            <input name="bank_name" id="f_bank_name" type="text" class="w-full border rounded p-2" required>
          </div>
          <div>
            <label class="block text-sm text-gray-700">Account Holder Name</label>
            <input name="account_holder_name" id="f_account_holder_name" type="text" class="w-full border rounded p-2" required>
          </div>
          <div>
            <label class="block text-sm text-gray-700">Account Number</label>
            <input name="bank_account_no" id="f_bank_account_no" type="text" class="w-full border rounded p-2" required>
          </div>
          <div>
            <label class="block text-sm text-gray-700">IFSC Code</label>
            <input name="IFSC_Code" id="f_IFSC_Code" type="text" class="w-full border rounded p-2">
          </div>
          <div>
            <label class="block text-sm text-gray-700">SWIFT Code</label>
            <input name="swift_code" id="f_swift_code" type="text" class="w-full border rounded p-2">
          </div>
          <div class="md:col-span-2">
            <label class="block text-sm text-gray-700">Bank Proof (image/pdf)</label>
            <input name="bank_detail" id="f_bank_detail" type="file" accept=".jpg,.jpeg,.png,.webp,.gif,.pdf" class="w-full border rounded p-2 bg-white">
            <p class="text-xs text-gray-500 mt-1">Uploading a new file replaces the existing proof.</p>
            <div id="currentProofWrap" class="mt-2 hidden">
              <span class="text-sm text-gray-700 mr-2">Current proof:</span>
              <a id="currentProofLink" href="#" target="_blank" class="text-blue-600 hover:underline text-sm">Open</a>
            </div>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="px-4 py-2 rounded bg-gray-200 hover:bg-gray-300" id="cancelBank">Cancel</button>
        <button type="submit" class="px-4 py-2 rounded bg-[#CD7F32] text-white hover:bg-[#b46e2b]">Save</button>
      </div>
    </form>
  </div>
</div>

<!-- Proof Viewer Modal -->
<div id="proofModal" class="modal">
  <div class="modal-inner">
    <div class="modal-header">
      <div class="modal-title">Bank Proof</div>
      <button class="close-btn" id="closeProofModal">&times;</button>
    </div>
    <div class="modal-body">
      <div id="proofContent" class="bg-white p-2 rounded">
        <!-- injected -->
      </div>
    </div>
    <div class="modal-footer">
      <button type="button" class="px-4 py-2 rounded bg-gray-200 hover:bg-gray-300" id="okProof">Close</button>
    </div>
  </div>
</div>

<?php include "footer.php";?>

<script>
(function(){
  const $ = (sel, root=document) => root.querySelector(sel);
  const $$ = (sel, root=document) => Array.from(root.querySelectorAll(sel));

  // ---- Search + Filter (unchanged) ----
  const searchInput = $('#search');
  const statusFilter = $('#status-filter');

  function filterTable(){
    const q = (searchInput?.value || '').toLowerCase().trim();
    const status = statusFilter?.value || ''; // '' | '1' | '0'
    $$('#bankTable tbody tr').forEach(tr => {
      const bankStatus = tr.getAttribute('data-bank-status') || '';
      const text = tr.innerText.toLowerCase();
      const matchText = q === '' || text.includes(q);
      const matchStatus = status === '' || status === bankStatus;
      tr.style.display = (matchText && matchStatus) ? '' : 'none';
    });
  }
  searchInput?.addEventListener('input', filterTable);
  statusFilter?.addEventListener('change', filterTable);

  // ---- Modal helpers ----
  const bankModal  = $('#bankModal');
  const proofModal = $('#proofModal');
  const openModal  = (m)=>{ if(m) m.style.display='flex'; };
  const closeModal = (m)=>{ if(m) m.style.display='none'; };

  $('#closeBankModal')?.addEventListener('click', () => closeModal(bankModal));
  $('#cancelBank')?.addEventListener('click', () => closeModal(bankModal));
  $('#closeProofModal')?.addEventListener('click', () => closeModal(proofModal));
  $('#okProof')?.addEventListener('click', () => closeModal(proofModal));
  window.addEventListener('keydown', (e)=>{ if(e.key==='Escape'){ closeModal(bankModal); closeModal(proofModal);} });
  [bankModal, proofModal].forEach(m => m?.addEventListener('click', e=>{ if(e.target===m) closeModal(m);} ));

  // ---- OPEN modal for Add/Update + View Proof (delegated on document) ----
  document.addEventListener('click', function(e){
    const editBtn = e.target.closest('.edit-bank');
    if (editBtn) {
      // Fill form
      $('#f_id').value = editBtn.getAttribute('data-id');
      const userLabel = `${editBtn.getAttribute('data-fullname')||''} <${editBtn.getAttribute('data-email')||''}>`;
      $('#f_user_label').value = userLabel;

      $('#f_bank_name').value = editBtn.getAttribute('data-bank_name') || '';
      $('#f_account_holder_name').value = editBtn.getAttribute('data-account_holder_name') || '';
      $('#f_bank_account_no').value = editBtn.getAttribute('data-bank_account_no') || '';
      $('#f_IFSC_Code').value = editBtn.getAttribute('data-IFSC_Code') || '';
      $('#f_swift_code').value = editBtn.getAttribute('data-swift_code') || '';

      const proof = editBtn.getAttribute('data-proof') || '';
      if (proof) {
        $('#currentProofWrap').classList.remove('hidden');
        $('#currentProofLink').href = proof;
      } else {
        $('#currentProofWrap').classList.add('hidden');
        $('#currentProofLink').removeAttribute('href');
      }
      openModal(bankModal);
      return;
    }

    const viewBtn = e.target.closest('.view-proof');
    if (viewBtn) {
      const src = viewBtn.getAttribute('data-src') || '';
      if (!src) return;
      const ext = src.split('.').pop().toLowerCase();
      const content = $('#proofContent');
      content.innerHTML = '';
      if (['jpg','jpeg','png','webp','gif'].includes(ext)) {
        const img = document.createElement('img');
        img.src = src; img.alt = 'Bank proof'; img.className = 'max-w-full h-auto rounded';
        content.appendChild(img);
      } else if (ext === 'pdf') {
        const iframe = document.createElement('iframe');
        iframe.src = src; iframe.className = 'w-full h-[70vh]'; content.appendChild(iframe);
      } else {
        const a = document.createElement('a');
        a.href = src; a.target = '_blank'; a.className = 'text-blue-600 hover:underline'; a.textContent = 'Open file';
        content.appendChild(a);
      }
      openModal(proofModal);
    }
  });

  // ---- Submit bank form (AJAX with file) ----
  $('#bankForm')?.addEventListener('submit', function(e){
    e.preventDefault();

    // small normalization: IFSC uppercase
    const ifsc = $('#f_IFSC_Code');
    if (ifsc) ifsc.value = (ifsc.value||'').toUpperCase().trim();

    const fd = new FormData(this);
    fetch(window.location.href, { method:'POST', body: fd })
      .then(r => r.ok ? r.json() : Promise.reject(new Error('Network error')))
      .then(res => {
        if (!res || !res.ok) throw new Error(res && res.error ? res.error : 'Unknown error');

        const id = String(res.row.id);
        const tr = Array.from(document.querySelectorAll('#bankTable tbody tr'))
          .find(r => r.firstElementChild && r.firstElementChild.textContent.trim() === id);
        if (tr) {
          tr.querySelector('.td-bank_name').textContent = res.row.bank_name || '';
          tr.querySelector('.td-account_holder_name').textContent = res.row.account_holder_name || '';
          tr.querySelector('.td-bank_account_no').textContent = res.row.bank_account_no || '';
          tr.querySelector('.td-IFSC_Code').textContent = res.row.IFSC_Code || '';
          tr.querySelector('.td-swift_code').textContent = res.row.swift_code || '';
          tr.setAttribute('data-bank-status','1');
          tr.querySelector('.td-status').innerHTML = '<span class="badge green">Updated</span>';

          const proofCell = tr.querySelector('.td-proof');
          const editBtn = tr.querySelector('.edit-bank');
          if (res.row.bank_detail) {
            proofCell.innerHTML = '<button type="button" class="px-3 py-1 rounded bg-blue-600 text-white text-sm hover:bg-blue-700 view-proof" data-src="'+res.row.bank_detail+'">View</button>';
            editBtn.setAttribute('data-proof', res.row.bank_detail);
          }
          editBtn.textContent = 'Update';
          editBtn.setAttribute('data-bank_name', res.row.bank_name || '');
          editBtn.setAttribute('data-account_holder_name', res.row.account_holder_name || '');
          editBtn.setAttribute('data-bank_account_no', res.row.bank_account_no || '');
          editBtn.setAttribute('data-IFSC_Code', res.row.IFSC_Code || '');
          editBtn.setAttribute('data-swift_code', res.row.swift_code || '');
        }
        closeModal(bankModal);
      })
      .catch(err => { alert('Save failed: ' + err.message); });
  });

  // initial filter
  filterTable();
})();
</script>
<script>
window.openBankModal = function(btn){
  const get = (a) => btn.getAttribute(a) || '';

  document.getElementById('f_id').value = get('data-id');
  document.getElementById('f_user_label').value =
    (get('data-fullname')||'') + ' <' + (get('data-email')||'') + '>';

  document.getElementById('f_bank_name').value = get('data-bank_name');
  document.getElementById('f_account_holder_name').value = get('data-account_holder_name');
  document.getElementById('f_bank_account_no').value = get('data-bank_account_no');
  document.getElementById('f_IFSC_Code').value = get('data-IFSC_Code');
  document.getElementById('f_swift_code').value = get('data-swift_code');

  const proof = get('data-proof');
  const wrap  = document.getElementById('currentProofWrap');
  const link  = document.getElementById('currentProofLink');
  if (proof) { wrap.classList.remove('hidden'); link.href = proof; }
  else { wrap.classList.add('hidden'); link.removeAttribute('href'); }

  document.getElementById('bankModal').style.display = 'flex';
};

window.openProofModal = function(btn){
  const src = btn.getAttribute('data-src') || '';
  if (!src) return;
  const ext = src.split('.').pop().toLowerCase();
  const content = document.getElementById('proofContent');
  content.innerHTML = '';
  if (['jpg','jpeg','png','webp','gif'].includes(ext)) {
    const img = document.createElement('img');
    img.src = src; img.alt = 'Bank proof'; img.className = 'max-w-full h-auto rounded';
    content.appendChild(img);
  } else if (ext === 'pdf') {
    const iframe = document.createElement('iframe');
    iframe.src = src; iframe.className = 'w-full h-[70vh]';
    content.appendChild(iframe);
  } else {
    const a = document.createElement('a');
    a.href = src; a.target = '_blank'; a.className = 'text-blue-600 hover:underline';
    a.textContent = 'Open file';
    content.appendChild(a);
  }
  document.getElementById('proofModal').style.display = 'flex';
};
</script>

</body>
</html>
