<?php
if (session_status() === PHP_SESSION_NONE) { session_start(); }
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include "config.php";

/* Ensure we have a PDO handle in $conn */
if (!isset($conn) || !($conn instanceof PDO)) {
  if (isset($dbh) && $dbh instanceof PDO) { $conn = $dbh; }
  else { die("No PDO connection found from config.php"); }
}

if (empty($_SESSION['clogin'])) {
  header('Location: login.php'); exit;
}

$email = $_SESSION['clogin'];

/* ---- Params ---- */
$per_page = isset($_GET['per_page']) ? max(1, (int)$_GET['per_page']) : 10;
$page     = isset($_GET['page'])     ? max(1, (int)$_GET['page'])     : 1;
$start    = ($page - 1) * $per_page;

$date_filter = isset($_GET['date_filter']) && $_GET['date_filter'] !== '' ? $_GET['date_filter'] : '';
$search      = isset($_GET['search']) ? trim($_GET['search']) : '';

/* ---- Base WHERE (only IB withdrawals from total_balance) ----
   trading_withdrawal is VARCHAR in your schema, so cast numerically.
*/
$where  = " WHERE email = :email
            AND CAST(COALESCE(trading_withdrawal,'0') AS DECIMAL(18,2)) > 0";

$params = [':email' => $email];

if ($date_filter !== '') {
  $where .= " AND DATE(reg_date) = :date_filter";
  $params[':date_filter'] = $date_filter;  // format 'YYYY-MM-DD'
}

if ($search !== '') {
  $where .= " AND (trade_id LIKE :search OR status LIKE :search OR deposit_type LIKE :search)";
  $params[':search'] = "%{$search}%";
}

/* ---- Count for pagination ---- */
$sql_count = "SELECT COUNT(*) FROM total_balance" . $where;
$stmt_cnt  = $conn->prepare($sql_count);
foreach ($params as $k => $v) { $stmt_cnt->bindValue($k, $v); }
$stmt_cnt->execute();
$total_rows  = (int)$stmt_cnt->fetchColumn();
$total_pages = (int)ceil($total_rows / $per_page);

/* ---- Data query ---- */
$sql = "SELECT id,
               trade_id,
               trading_withdrawal AS amount,
               status,
               deposit_type,
               reg_date
        FROM total_balance
        $where
        ORDER BY reg_date DESC, id DESC
        LIMIT :start, :per_page";
$stmt = $conn->prepare($sql);

foreach ($params as $k => $v) { $stmt->bindValue($k, $v); }
$stmt->bindValue(':start',    (int)$start,    PDO::PARAM_INT);
$stmt->bindValue(':per_page', (int)$per_page, PDO::PARAM_INT);

$stmt->execute();
$withdraws = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* Helpers for preserving query string on pagination/filters */
function qs($over = []) {
  $q = array_merge($_GET, $over);
  return '?' . http_build_query($q);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
  <title>IB Withdraw Report</title>
  <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
</head>
<body class="bg-gray-100">
  <?php include "header.php"; ?>

  <div class="flex flex-col md:flex-row min-h-screen">
    <!-- Sidebar -->
    <?php include "side_bar.php"; ?>

    <!-- Main Content -->
    <main class="flex-1 p-4 md:p-6 md:ml-64" style="width:100%;">
      <h1 class="text-2xl md:text-3xl font-bold text-center mb-4 text-[#CD7F32]">IB Withdraw Report</h1>

      <!-- Filters -->
      <div class="mb-4 grid gap-3 md:grid-cols-2">
        <form method="GET" class="flex items-center space-x-2">
          <input type="hidden" name="per_page" value="<?= htmlspecialchars($per_page) ?>">
          <label class="text-sm text-[#CD7F32] font-semibold">Date:</label>
          <input type="date" name="date_filter" value="<?= htmlspecialchars($date_filter) ?>" class="px-2 py-1 border border-gray-300 rounded text-sm">
          <button type="submit" class="px-3 py-1 bg-[#CD7F32] text-white text-sm rounded">Filter</button>
          <?php if ($date_filter !== '' || $search !== ''): ?>
            <a href="<?= htmlspecialchars(basename(__FILE__)) ?>" class="px-3 py-1 bg-gray-200 text-gray-800 text-sm rounded">Clear</a>
          <?php endif; ?>
        </form>

        <form method="GET" class="flex items-center space-x-2">
          <input type="hidden" name="per_page" value="<?= htmlspecialchars($per_page) ?>">
          <input type="text" name="search" value="<?= htmlspecialchars($search) ?>"
                 placeholder="Search Trade ID / Status / Type"
                 class="px-2 py-1 border border-gray-300 rounded text-sm" />
          <button type="submit" class="px-3 py-1 bg-[#CD7F32] text-white text-sm rounded">Search</button>
        </form>
      </div>

      <!-- Records Per Page -->
      <form method="GET" class="mb-4 flex items-center space-x-2">
        <?php
          // keep filters when changing per_page
          if ($date_filter !== '') echo '<input type="hidden" name="date_filter" value="'.htmlspecialchars($date_filter).'">';
          if ($search !== '')      echo '<input type="hidden" name="search" value="'.htmlspecialchars($search).'">';
          if ($page)               echo '<input type="hidden" name="page" value="1">'; // reset page when per_page changes
        ?>
        <label class="text-sm text-[#CD7F32] font-semibold">Records per page:</label>
        <select name="per_page" class="px-2 py-1 border border-gray-300 rounded text-sm" onchange="this.form.submit()">
          <option value="10" <?= $per_page==10?'selected':''; ?>>10</option>
          <option value="25" <?= $per_page==25?'selected':''; ?>>25</option>
          <option value="50" <?= $per_page==50?'selected':''; ?>>50</option>
        </select>
      </form>

      <!-- Withdraw Table -->
      <div class="overflow-x-auto">
        <table class="w-full bg-white border border-gray-200 rounded-lg shadow-md text-sm">
          <thead>
            <tr class="bg-gray-100 text-left text-[#CD7F32]">
              <th class="px-2 py-1 border-b">ID</th>
              <th class="px-2 py-1 border-b">Trade ID (MT5)</th>
              <th class="px-2 py-1 border-b">Amount</th>
              <th class="px-2 py-1 border-b">Status</th>
              <th class="px-2 py-1 border-b">Type</th>
              <th class="px-2 py-1 border-b">Date</th>
            </tr>
          </thead>
          <tbody>
            <?php if ($withdraws): foreach ($withdraws as $w): ?>
              <tr class="hover:bg-gray-50">
                <td class="px-2 py-1 border-b"><?= htmlspecialchars($w['id']) ?></td>
                <td class="px-2 py-1 border-b"><?= htmlspecialchars($w['trade_id']) ?></td>
                <td class="px-2 py-1 border-b">
                  <?= htmlspecialchars(number_format((float)$w['amount'], 2)) ?>
                </td>
                <td class="px-2 py-1 border-b"><?= htmlspecialchars($w['status'] ?? '') ?></td>
                <td class="px-2 py-1 border-b"><?= htmlspecialchars($w['deposit_type'] ?? '') ?></td>
                <td class="px-2 py-1 border-b"><?= htmlspecialchars($w['reg_date']) ?></td>
              </tr>
            <?php endforeach; else: ?>
              <tr><td colspan="6" class="px-2 py-3 text-center text-gray-600">No IB withdrawals found.</td></tr>
            <?php endif; ?>
          </tbody>
        </table>
      </div>

      <!-- Pagination -->
      <div class="mt-4 flex flex-col sm:flex-row justify-between items-center space-y-3 sm:space-y-0">
        <span class="text-xs text-gray-700">Page <?= $page ?> of <?= max(1,$total_pages) ?></span>
        <div class="flex space-x-2">
          <?php if ($page > 1): ?>
            <a href="<?= qs(['page'=>$page-1]) ?>" class="px-3 py-1 bg-[#CD7F32] text-white text-xs rounded">Previous</a>
          <?php endif; ?>
          <?php if ($page < $total_pages): ?>
            <a href="<?= qs(['page'=>$page+1]) ?>" class="px-3 py-1 bg-[#CD7F32] text-white text-xs rounded">Next</a>
          <?php endif; ?>
        </div>
      </div>

      <br><br><br>
      <?php include "footer.php"; ?>
    </main>
  </div>
</body>
</html>
