<?php
// news.php — list + view news; mark-as-read only when "View" is clicked

if (session_status() === PHP_SESSION_NONE) { session_start(); }
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
date_default_timezone_set('Asia/Kolkata');

require_once __DIR__ . '/config.php';

// Normalize PDO handle
if (!isset($conn) || !($conn instanceof PDO)) {
  if (isset($dbh) && $dbh instanceof PDO) { $conn = $dbh; }
  else { die('No PDO connection available from config.php'); }
}

if (empty($_SESSION['clogin'])) {
  header('Location: login.php'); exit;
}
$userEmail = $_SESSION['clogin'];

// Helpers
function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
function qs($over = []) { $q = array_merge($_GET, $over); return '?' . http_build_query($q); }

// Actions
$view_id   = isset($_GET['view_id']) && ctype_digit($_GET['view_id']) ? (int)$_GET['view_id'] : null;
// also support legacy pattern: ?id=123&view=1
if (!$view_id && isset($_GET['id']) && ctype_digit($_GET['id']) && isset($_GET['view'])) {
  $view_id = (int)$_GET['id'];
}
$action    = $_GET['action'] ?? '';

// Mark all as read (explicit action only)
if ($action === 'mark_all') {
  $sql = "INSERT IGNORE INTO news_reads (news_id, user_email)
          SELECT n.id, :email FROM news n";
  $st = $conn->prepare($sql);
  $st->execute([':email' => $userEmail]);
  header('Location: news.php'); exit;
}

// If a specific item is being viewed, fetch it and mark as read
$viewItem = null;
if ($view_id) {
  // Load item
  $s = $conn->prepare("SELECT id, title, content, created_at, `date` FROM news WHERE id = :id");
  $s->execute([':id' => $view_id]);
  $viewItem = $s->fetch(PDO::FETCH_ASSOC);

  if ($viewItem) {
    // Mark as read for this user (only now, on view)
    $mr = $conn->prepare("
      INSERT INTO news_reads (news_id, user_email, read_at)
      VALUES (:nid, :email, NOW())
      ON DUPLICATE KEY UPDATE read_at = VALUES(read_at)
    ");
    $mr->execute([':nid' => $view_id, ':email' => $userEmail]);
  }
}

// List filters & pagination
$per_page = max(1, (int)($_GET['per_page'] ?? 10));
$page     = max(1, (int)($_GET['page'] ?? 1));
$start    = ($page - 1) * $per_page;
$search   = trim($_GET['search'] ?? '');

// Build WHERE (search by title/content)
$where  = " WHERE 1 ";
$params = [];
if ($search !== '') {
  $where .= " AND (n.title LIKE :q OR n.content LIKE :q) ";
  $params[':q'] = "%{$search}%";
}

// Count total
$sqlCount = "SELECT COUNT(*) FROM news n" . $where;
$stc = $conn->prepare($sqlCount);
foreach ($params as $k=>$v) { $stc->bindValue($k, $v); }
$stc->execute();
$total_rows  = (int)$stc->fetchColumn();
$total_pages = (int)ceil($total_rows / $per_page);

// Fetch page (with read status)
$sql = "
  SELECT n.id, n.title, n.created_at, n.`date`,
         CASE WHEN r.news_id IS NULL THEN 0 ELSE 1 END AS is_read
  FROM news n
  LEFT JOIN news_reads r
    ON r.news_id = n.id AND r.user_email = :email
  $where
  ORDER BY n.created_at DESC, n.id DESC
  LIMIT :start, :per_page
";
$st = $conn->prepare($sql);
$st->bindValue(':email', $userEmail);
foreach ($params as $k=>$v) { $st->bindValue($k, $v); }
$st->bindValue(':start', (int)$start, PDO::PARAM_INT);
$st->bindValue(':per_page', (int)$per_page, PDO::PARAM_INT);
$st->execute();
$list = $st->fetchAll(PDO::FETCH_ASSOC);

?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8"/>
  <meta name="viewport" content="width=device-width,initial-scale=1"/>
  <title>News</title>
  <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
</head>
<body class="bg-gray-50">
  <?php include 'header.php'; ?>

  <div class="flex flex-col md:flex-row min-h-screen">
    <?php if (file_exists(__DIR__ . '/side_bar.php')) include 'side_bar.php'; ?>

    <main class="flex-1 p-4 md:p-6 md:ml-64" style="width:100%;">
      <div class="flex items-center justify-between mb-4">
        <h1 class="text-2xl md:text-3xl font-bold text-[#CD7F32]">News</h1>
        <div class="flex items-center gap-2">
          <form method="get" class="flex items-center gap-2">
            <input type="text" name="search" value="<?= h($search) ?>"
                   placeholder="Search title / content"
                   class="px-3 py-2 border border-gray-300 rounded text-sm bg-white"/>
            <select name="per_page" class="px-2 py-2 border border-gray-300 rounded text-sm bg-white" onchange="this.form.submit()">
              <option value="10" <?= $per_page==10?'selected':''; ?>>10</option>
              <option value="25" <?= $per_page==25?'selected':''; ?>>25</option>
              <option value="50" <?= $per_page==50?'selected':''; ?>>50</option>
            </select>
            <button class="px-3 py-2 bg-[#CD7F32] text-white rounded text-sm">Apply</button>
            <?php if ($search !== ''): ?>
              <a href="news.php" class="px-3 py-2 bg-gray-200 rounded text-sm">Clear</a>
            <?php endif; ?>
          </form>
        
        </div>
      </div>

      <!-- Detail panel (only when viewing a single item) -->
      <?php if ($viewItem): ?>
        <div class="bg-white border border-gray-200 rounded-lg shadow mb-6">
          <div class="px-4 py-3 border-b border-gray-200 flex items-center justify-between">
            <h2 class="text-lg font-bold"><?= h($viewItem['title']) ?></h2>
            <a href="news.php" class="text-sm text-blue-600 hover:underline">Back to list</a>
          </div>
          <div class="px-4 py-4 prose max-w-none">
            <?php
              // If your content is trusted HTML, echo as-is; otherwise escape:
              // echo nl2br(h($viewItem['content']));
              echo $viewItem['content']; // assuming content saved as safe HTML from admin
            ?>
          </div>
          <div class="px-4 py-3 border-t border-gray-200 text-xs text-gray-500">
            Created: <?= h($viewItem['created_at'] ?? '') ?> &bull; Date: <?= h($viewItem['date'] ?? '') ?>
          </div>
        </div>
      <?php endif; ?>

      <!-- List -->
      <div class="bg-white border border-gray-200 rounded-lg shadow">
        <div class="px-4 py-3 border-b border-gray-200 flex items-center justify-between">
          <div class="font-semibold">All News</div>
          <div class="text-xs text-gray-500">Page <?= $page ?> of <?= max(1,$total_pages) ?></div>
        </div>

        <div class="overflow-x-auto">
          <table class="w-full text-sm">
            <thead>
              <tr class="bg-gray-100 text-left text-[#CD7F32]">
                <th class="px-3 py-2 border-b">Title</th>
                <th class="px-3 py-2 border-b">Created</th>
                <th class="px-3 py-2 border-b">Status</th>
                <th class="px-3 py-2 border-b">Action</th>
              </tr>
            </thead>
            <tbody>
              <?php if ($list): foreach ($list as $row): ?>
                <tr class="hover:bg-gray-50">
                  <td class="px-3 py-2 border-b">
                    <span class="font-medium"><?= h($row['title']) ?></span>
                  </td>
                  <td class="px-3 py-2 border-b"><?= h($row['created_at']) ?></td>
                  <td class="px-3 py-2 border-b">
                    <?php if ((int)$row['is_read'] === 1): ?>
                      <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-semibold bg-green-100 text-green-700">Read</span>
                    <?php else: ?>
                      <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-semibold bg-red-100 text-red-700">New</span>
                    <?php endif; ?>
                  </td>
                  <td class="px-3 py-2 border-b">
                    <a href="<?= h('news.php?view_id='.(int)$row['id']) ?>"
                       class="px-3 py-1 bg-[#CD7F32] text-white rounded text-xs">
                      View
                    </a>
                  </td>
                </tr>
              <?php endforeach; else: ?>
                <tr>
                  <td colspan="4" class="px-3 py-6 text-center text-gray-600">No news found.</td>
                </tr>
              <?php endif; ?>
            </tbody>
          </table>
        </div>

        <!-- Pagination -->
        <div class="px-4 py-3 border-t border-gray-200 flex items-center justify-between">
          <div class="text-xs text-gray-600">
            Showing <?= $total_rows ? ($start+1) : 0 ?>–<?= min($start+$per_page, $total_rows) ?> of <?= $total_rows ?>
          </div>
          <div class="flex gap-2">
            <?php if ($page > 1): ?>
              <a class="px-3 py-1 bg-[#CD7F32] text-white rounded text-xs" href="<?= h(qs(['page'=>$page-1])) ?>">Previous</a>
            <?php endif; ?>
            <?php if ($page < $total_pages): ?>
              <a class="px-3 py-1 bg-[#CD7F32] text-white rounded text-xs" href="<?= h(qs(['page'=>$page+1])) ?>">Next</a>
            <?php endif; ?>
          </div>
        </div>
      </div>

      <br><br><br>
      <?php include 'footer.php'; ?>
    </main>
  </div>
</body>
</html>
