<?php
/* accounts_grid.php — with fixed Header + Sidebar layout */
declare(strict_types=1);
if (session_status() !== PHP_SESSION_ACTIVE) session_start();

require_once __DIR__.'/config.php';
require_once __DIR__.'/manager_initialize.php'; // defines $api and MTRetCode

$email = $_SESSION['clogin'] ?? '';
if (!$email) { echo "<script>location.href='login.php';</script>"; exit; }

/* -------- Inputs -------- */
$from = isset($_GET['from']) && $_GET['from'] !== '' ? $_GET['from'] : null; // YYYY-MM-DD
$to   = isset($_GET['to'])   && $_GET['to']   !== '' ? $_GET['to']   : null;
$qs   = isset($_GET['q'])    ? trim($_GET['q']) : ''; // for prefilled search box

/* -------- Query liveaccount with optional date filter -------- */
$sql = "SELECT id, trade_id, account_name, Registered_Date FROM liveaccount WHERE email = :e";
$params = [':e'=>$email];
if ($from) { $sql .= " AND DATE(Registered_Date) >= :from"; $params[':from'] = $from; }
if ($to)   { $sql .= " AND DATE(Registered_Date) <= :to";   $params[':to']   = $to;   }
$sql .= " ORDER BY id DESC";

$stmt = $conn->prepare($sql);
$stmt->execute($params);
$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* -------- Build cards from MT5 Manager data -------- */
$cards = [];
$totalBalance = 0.0;
foreach ($rows as $r) {
  $login = (int)($r['trade_id'] ?? 0);
  if ($login <= 0) continue;

  $name  = '—';
  $group = '—';
  $bal   = 0.0;

  // MT5 profile
  $ret = $api->UserGet($login, $user);
  if ($ret === MTRetCode::MT_RET_OK && $user) {
    $name  = $user->Name ?? $user->NameFull ?? '—';
    $group = $user->Group ?? '—';
  }

  // MT5 money snapshot
  $ret = $api->UserAccountGet($login, $acct);
  if ($ret === MTRetCode::MT_RET_OK && $acct) {
    $bal = (float)($acct->Balance ?? 0);
  }

  $totalBalance += $bal;

  $cards[] = [
    'login'      => (string)$login,
    'name'       => (string)$name,
    'group'      => (string)$group,
    'balance'    => number_format($bal, 2),
    'registered' => $r['Registered_Date'] ? date('Y-m-d', strtotime($r['Registered_Date'])) : ''
  ];
}
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Your MT5 Accounts</title>
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">

<style>
  :root{
    /* Layout sizes (tweak to match your actual header/sidebar) */
    --header-h: 64px;            /* fixed header height */
    --sidebar-w: 260px;          /* desktop sidebar width */
    --sidebar-w-sm: 0px;         /* collapsed on small screens */

    /* Theme */
    --o1:#ff9800; /* main orange */
    --o2:#f57c00; /* deep orange */
    --ink:#1f2937;
    --muted:#6b7280;
    --line:#eceff3;
    --card:#ffffff;
    --bg:#fafafa;
  }

  *{box-sizing:border-box}
  body{
    margin:0;
    font-family:system-ui,-apple-system,Segoe UI,Roboto,Inter,Arial,sans-serif;
    color:var(--ink);
    background:var(--bg);
  }

  /* ====== FIXED HEADER (wrap your header.php inside) ====== */
  .app-header{
    position:fixed; inset:0 0 auto 0; height:var(--header-h);
    background:#fff; border-bottom:1px solid var(--line);
    z-index:1000; display:flex; align-items:center;
  }
  .app-header > .inner{ width:100%; height:100%; overflow:hidden; }

  /* ====== FIXED SIDEBAR (wrap side_bar.php) ====== */
  .app-sidebar{
    position:fixed; top:var(--header-h); bottom:0; left:0;
    width:var(--sidebar-w); background:#fff;
    border-right:1px solid var(--line); overflow:auto; z-index:900;
  }
  /* Mobile: sidebar off-canvas by default */
  @media (max-width: 900px){
    .app-sidebar{ width:var(--sidebar-w); transform:translateX(-100%); transition:transform .25s ease; }
    .app-sidebar.open{ transform:translateX(0); }
  }

  /* ====== MAIN CONTENT (never overlap) ====== */
  .app-main{
    padding-top:calc(var(--header-h) + 16px);
    padding-left:16px; padding-right:16px;
    margin-left:var(--sidebar-w);
  }
  @media (max-width: 900px){
    .app-main{ margin-left:var(--sidebar-w-sm); }
  }

  .wrap{max-width:1120px;margin:0 auto;}

  /* Hero summary */
  .hero{
    border-radius:18px;
    padding:22px;
    background:
      radial-gradient(80% 160% at 110% -20%, rgba(255,255,255,.35) 0%, transparent 60%),
      linear-gradient(135deg, var(--o1), var(--o2));
    color:#fff;
    box-shadow:0 14px 40px rgba(245,124,0,.25);
    display:flex; align-items:center; gap:20px; justify-content:space-between; flex-wrap:wrap;
  }
  .hero .title{font-size:1.6rem;font-weight:800;margin:0}
  .pill{
    background:rgba(255,255,255,.18);
    border:1px solid rgba(255,255,255,.35);
    color:#fff;
    padding:8px 14px;
    border-radius:999px;
    font-weight:700;
    display:inline-flex; align-items:center; gap:8px;
  }

  /* Filters */
  .filters{
    margin:18px 0 6px;
    background:#fff;
    border:1px solid var(--line);
    border-radius:14px;
    padding:12px;
    display:grid;
    grid-template-columns: 1fr;
    gap:10px;
    box-shadow:0 8px 24px rgba(0,0,0,.05);
  }
  @media (min-width:700px){
    .filters{grid-template-columns: repeat(12, 1fr); align-items:end}
  }
  .field{display:flex; flex-direction:column; gap:6px}
  .field label{font-size:.9rem; color:var(--muted)}
  .input, .btn{
    height:40px; border-radius:10px; border:1px solid var(--line);
    padding:0 12px; font-size:14px; outline:none;
  }
  .input:focus{border-color:#ffc78a; box-shadow:0 0 0 3px rgba(255,152,0,.15)}
  .btn{cursor:pointer; font-weight:700}
  .btn-orange{background:linear-gradient(135deg,var(--o1),var(--o2)); color:#fff; border:none; box-shadow:0 8px 20px rgba(245,124,0,.25)}
  .btn-ghost{background:#fff; color:var(--o2); border:1px solid #ffd5a6}
  .btn-ghost:hover{background:#fff7e9}
  .muted{color:var(--muted)}
  .right{justify-self:end}

  /* Grid */
  .grid{
    margin-top:14px;
    display:grid;
    grid-template-columns:repeat(12,1fr);
    gap:18px;
  }

  /* Card */
  .acc-card{
    grid-column:span 12;
    background:var(--card);
    border:1px solid var(--line);
    border-radius:16px;
    overflow:hidden;
    box-shadow:0 10px 28px rgba(0,0,0,.06);
  }
  @media (min-width:720px){ .acc-card{grid-column:span 6} }
  @media (min-width:1024px){ .acc-card{grid-column:span 4} }

  .acc-top{
    background: linear-gradient(135deg, var(--o1), var(--o2));
    color:#fff;
    padding:14px 16px;
    display:flex; align-items:center; justify-content:space-between; gap:12px;
  }
  .badge{
    display:inline-flex; align-items:center; gap:8px;
    padding:6px 10px; border-radius:999px;
    background:rgba(255,255,255,.18);
    border:1px solid rgba(255,255,255,.35);
    font-weight:600;
  }
  .acc-body{padding:16px 16px; display:grid; grid-template-columns:1fr; gap:12px}
  .row{display:flex; align-items:center; justify-content:space-between; gap:8px}
  .label{color:var(--muted); font-size:.92rem}
  .value{font-weight:700; font-size:1.06rem}
  .mono{font-family: ui-monospace,SFMono-Regular,Menlo,Monaco,Consolas,monospace}
  .balance{
    display:flex; align-items:center; justify-content:space-between; gap:12px;
    padding:12px 14px; border-radius:14px; border:1px dashed var(--line);
    background:linear-gradient(180deg,#fff, #fff9f0);
  }
  .amt{font-size:1.2rem; font-weight:800; color:#0f172a}
  .mini-actions{display:flex; gap:10px; flex-wrap:wrap}
  .copy{cursor:pointer}
  .no-data{grid-column:span 12; padding:18px; background:#fff; border:1px dashed var(--line); border-radius:14px}
  .results-bar{display:flex; gap:10px; align-items:center; margin-top:8px}

  /* Small “hamburger” to open sidebar on mobile */
  .hamb{ position:fixed; top:10px; left:10px; z-index:1100;
         background:linear-gradient(135deg,var(--o1),var(--o2)); color:#fff;
         border:none; border-radius:10px; padding:8px 10px; display:none; }
  @media (max-width:900px){ .hamb{ display:block; } }
</style>
</head>
<body>
<style>/* Two-column layout inside card body */
.row.two-col {
  display: grid;
  grid-template-columns: 1fr 1fr;
  gap: 12px;        /* space between columns */
}

@media (max-width: 600px) {
  .row.two-col {
    grid-template-columns: 1fr; /* stack on small screens */
  }
}
</style>
  <!-- Mobile sidebar toggle -->
  <button class="hamb" type="button" onclick="toggleSidebar()">
    <i class="fa-solid fa-bars"></i>
  </button>

  <!-- ===== Header (fixed) ===== -->
  <header class="app-header">
    <div class="inner">
      <?php include __DIR__."/header.php"; ?>
    </div>
  </header>

  <!-- ===== Sidebar (fixed) ===== -->
  <aside id="AppSidebar" class="app-sidebar">
    <?php include __DIR__."/side_bar.php"; ?>
  </aside>

  <!-- ===== Main content (spaced by header+sidebar) ===== -->
  <main class="app-main">
    <div class="wrap">
<br>
      <!-- Hero -->
      <section class="hero">
        <div>
          <div class="pill"><i class="fa-solid fa-wallet"></i> Live MT5 Accounts</div>
       </div>
        <div>
          <div class="pill"><i class="fa-solid fa-indian-rupee-sign"></i>
            Total Balance&nbsp;<strong><?php echo number_format($totalBalance, 2); ?></strong>
          </div>
        </div>
      </section>

      <!-- Filters -->
      <form class="filters" method="get" action="">
        <div class="field" style="grid-column:span 4;">
          <label for="from">From date</label>
          <input type="date" name="from" id="from" class="input" value="<?php echo htmlspecialchars((string)$from ?? ''); ?>">
        </div>
        <div class="field" style="grid-column:span 4;">
          <label for="to">To date</label>
          <input type="date" name="to" id="to" class="input" value="<?php echo htmlspecialchars((string)$to ?? ''); ?>">
        </div>
        <div class="field" style="grid-column:span 4;">
          <label for="q">Live search (Name / Trade ID / Type)</label>
          <input type="text" name="q" id="q" class="input" placeholder="Type to filter…" value="<?php echo htmlspecialchars($qs); ?>">
        </div>
        <div class="field right" style="grid-column:span 12; display:flex; gap:10px; justify-content:flex-end;">
          <button class="btn btn-ghost" type="button" onclick="resetFilters()">Reset</button>
          <button class="btn btn-orange" type="submit" title="Filter by date on server">Apply Date Filter</button>
        </div>
      </form>

      <div class="results-bar">
        <span class="muted"><i class="fa-regular fa-clock"></i>
          <?php
            $range = [];
            if ($from) $range[] = "from <strong>".htmlspecialchars($from)."</strong>";
            if ($to)   $range[] = "to <strong>".htmlspecialchars($to)."</strong>";
            echo $range ? 'Server range: '.implode(' ', $range) : 'No date range applied';
          ?>
        </span>
        <span id="visibleCount" class="muted"></span>
      </div>

      <!-- Cards -->
      <section id="cardsGrid" class="grid">
        <?php if (!$cards): ?>
          <div class="no-data">
            <strong>No accounts found</strong>
            <div class="muted">Try changing the date range or clear filters.</div>
          </div>
        <?php else: ?>
          <?php foreach ($cards as $c): ?>
           <article class="acc-card"
  data-name="<?php echo htmlspecialchars(mb_strtolower($c['name'])); ?>"
  data-login="<?php echo htmlspecialchars($c['login']); ?>"
  data-group="<?php echo htmlspecialchars(mb_strtolower($c['group'])); ?>">

  <header class="acc-top">
    <span class="badge"><i class="fa-solid fa-user-shield"></i> <?php echo htmlspecialchars($c['group']); ?></span>
    <span class="badge"><i class="fa-solid fa-hashtag"></i> <?php echo htmlspecialchars($c['login']); ?></span>
  </header>

  <div class="acc-body">
    <!-- Row 1: Name + Trade ID -->
    <div class="row two-col">
      <div>
        <div class="label">Account Holder</div>
        <div class="value"><?php echo htmlspecialchars($c['name']); ?></div>
      </div>
      <div>
        <div class="label">Trade ID</div>
        <div class="value mono copy" title="Click to copy"
             onclick="copyTxt('<?php echo htmlspecialchars($c['login']); ?>')">
          <?php echo htmlspecialchars($c['login']); ?> <i class="fa-regular fa-copy"></i>
        </div>
      </div>
    </div>

    <!-- Row 2: Account Type + Registered -->
    <div class="row two-col">
      <div>
        <div class="label">Account Type</div>
        <div class="value"><?php echo htmlspecialchars($c['group']); ?></div>
      </div>
      <div>
        <div class="label">Registered</div>
        <div class="value"><?php echo htmlspecialchars($c['registered']); ?></div>
      </div>
    </div>

    <!-- Balance -->
    <div class="balance">
      <div class="muted"><i class="fa-solid fa-coins"></i> Balance</div>
      <div class="amt">₹ <?php echo $c['balance']; ?></div>
    </div>

    <!-- Actions -->
    <div class="mini-actions">
      <a class="btn btn-orange" href="account_details.php?tradeid=<?php echo urlencode($c['login']); ?>">
        <i class="fa-solid fa-circle-info"></i> View Details
      </a>
      <!--<button class="btn btn-ghost" type="button" onclick="copyTxt('<?php echo htmlspecialchars($c['login']); ?>')">-->
      <!--  <i class="fa-regular fa-copy"></i> Copy ID-->
      <!--</button>-->
    </div>
  </div>
</article>

          <?php endforeach; ?>
        <?php endif; ?>
      </section>

    </div>
  </main>

<script>
/* Sidebar toggle for mobile */
function toggleSidebar(){
  const el = document.getElementById('AppSidebar');
  el.classList.toggle('open');
}

/* Copy helper */
function copyTxt(txt){
  navigator.clipboard.writeText(txt).then(()=> {
    const prev = document.title;
    document.title = 'Copied!';
    setTimeout(()=>document.title = prev, 600);
  });
}

/* Live search across name/login/group */
(function(){
  const q = document.getElementById('q');
  const grid = document.getElementById('cardsGrid');
  const cards = Array.from(grid.querySelectorAll('.acc-card'));
  const counter = document.getElementById('visibleCount');

  function applyFilter(){
    const term = (q.value || '').trim().toLowerCase();
    let visible = 0;
    cards.forEach(card => {
      const name  = card.dataset.name || '';
      const login = card.dataset.login || '';
      const group = card.dataset.group || '';
      const show = !term || name.includes(term) || login.includes(term) || group.includes(term);
      card.style.display = show ? '' : 'none';
      if (show) visible++;
    });
    counter.innerHTML = visible + " matches shown";
  }

  q.addEventListener('input', applyFilter);
  applyFilter(); // run once
})();

/* Reset filters (client + server) */
function resetFilters(){
  const url = new URL(window.location.href);
  url.searchParams.delete('from');
  url.searchParams.delete('to');
  url.searchParams.delete('q');
  window.location.href = url.toString();
}
</script>
</body>
</html>
