<?php
/* ---------------------------------------------------------
   TRADES — ORANGE/WHITE UI with Datewise Filter
   Source table: postion_record_history
   Filter: WHERE email = :eid (session 'clogin')
   Date filter logic: DATE(COALESCE(NULLIF(close_time,''), open_time))
---------------------------------------------------------- */
session_start();
include 'config.php';

if (empty($_SESSION['clogin'])) {
  header('location:login.php'); exit;
}

// Optional debug: add ?debug=1
if (isset($_GET['debug'])) { ini_set('display_errors',1); error_reporting(E_ALL); } else { error_reporting(0); }

/* ====== SCHEMA MAP (edit if your column names differ) ====== */
$TABLE = 'postion_record_history';
$COL_ID        = 'id';           // if present (not displayed)
$COL_TRADE_ID  = 'trade_id';
$COL_EMAIL     = 'email';
$COL_GROUP     = 'campaigns';    // maps to account type
$COL_SYMBOL    = 'symbol';
$COL_ACTION    = 'action';       // BUY/SELL
$COL_VOLUME    = 'vol';
$COL_OPEN      = 'open_time';
$COL_CLOSE     = 'close_time';
$COL_PROFIT    = 'profit';
/* ============================================================ */

// Backtick helper
function ident($s){ return '`'.str_replace('`','``',$s).'`'; }
$T  = ident($TABLE);
$E  = ident($COL_EMAIL);
$TR = ident($COL_TRADE_ID);
$GR = ident($COL_GROUP);
$SY = ident($COL_SYMBOL);
$AC = ident($COL_ACTION);
$VO = ident($COL_VOLUME);
$OP = ident($COL_OPEN);
$CL = ident($COL_CLOSE);
$PF = ident($COL_PROFIT);

// Normalize PDO handle regardless of variable name in config.php
$pdo = null;
if (isset($dbh) && $dbh instanceof PDO) { $pdo = $dbh; }
elseif (isset($conn) && $conn instanceof PDO) { $pdo = $conn; }
if (!$pdo) { die('Database connection not found. Ensure config.php sets $dbh or $conn as PDO.'); }
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

// current user
$eid = $_SESSION['clogin'];

// Read date filter (YYYY-MM-DD)
$from = trim($_GET['from'] ?? '');
$to   = trim($_GET['to'] ?? '');
// Canonical: if only one provided, treat it as same-day range
if ($from && !$to) $to = $from;
if ($to && !$from) $from = $to;

// Build WHERE
$where = ["$E = :eid"];
$params = [':eid' => $eid];

// We filter on DATE(COALESCE(NULLIF(close_time,''), open_time))
$dateExpr = "DATE(COALESCE(NULLIF($CL,''), $OP))";
if ($from && $to) {
  $where[] = "$dateExpr BETWEEN :dfrom AND :dto";
  $params[':dfrom'] = $from;
  $params[':dto']   = $to;
}

// Fetch trades for this user with optional date filter
try {
  $sql = "SELECT $TR AS trade_id, $GR AS campaigns, $SY AS symbol, $AC AS action, $VO AS vol,
                 $OP AS open_time, $CL AS close_time, $PF AS profit
          FROM $T
          WHERE " . implode(' AND ', $where) . "
          ORDER BY $OP DESC";
  $q = $pdo->prepare($sql);
  foreach ($params as $k=>$v){ $q->bindValue($k, $v, PDO::PARAM_STR); }
  $q->execute();
  $rows = $q->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
  $rows = [];
}

// KPIs (unfiltered; keep as-is or make filtered by date if needed)
function kpi_sum_today($pdo,$T,$E,$eid,$PF,$CL,$OP){
  try {
    $sql = "SELECT COALESCE(SUM($PF),0) s
            FROM $T
            WHERE $E=:eid
              AND (
                    (LENGTH($CL)>0 AND DATE($CL)=CURDATE())
                 OR ( (LENGTH($CL)=0 OR $CL IS NULL) AND LENGTH($OP)>0 AND DATE($OP)=CURDATE() )
              )";
    $st = $pdo->prepare($sql);
    $st->bindParam(':eid',$eid);
    $st->execute();
    return (float)$st->fetchColumn();
  } catch (Throwable $e){ return 0.0; }
}
function kpi_win_rate_30d($pdo,$T,$E,$eid,$PF,$CL,$OP){
  try {
    $sql = "SELECT
              SUM(CASE WHEN $PF>0 THEN 1 ELSE 0 END) wins,
              COUNT(*) total
            FROM $T
            WHERE $E=:eid
              AND (
                    (LENGTH($CL)>0 AND $CL >= (NOW() - INTERVAL 30 DAY))
                 OR ( (LENGTH($CL)=0 OR $CL IS NULL) AND LENGTH($OP)>0 AND $OP >= (NOW() - INTERVAL 30 DAY))
              )";
    $st=$pdo->prepare($sql); $st->bindParam(':eid',$eid); $st->execute();
    $r=$st->fetch(PDO::FETCH_ASSOC);
    $t = (int)($r['total']??0); $w=(int)($r['wins']??0);
    return $t? round(($w/$t)*100):0;
  } catch (Throwable $e){ return 0; }
}
function kpi_open_trades_guess($pdo,$T,$E,$eid,$CL){
  try {
    $sql="SELECT COUNT(*) FROM $T WHERE $E=:eid AND ( $CL IS NULL OR $CL='' OR $CL='0000-00-00 00:00:00')";
    $st=$pdo->prepare($sql); $st->bindParam(':eid',$eid); $st->execute();
    return (int)$st->fetchColumn();
  } catch (Throwable $e){ return 0; }
}
function kpi_closed_this_month($pdo,$T,$E,$eid,$CL){
  try {
    $sql="SELECT COUNT(*) FROM $T WHERE $E=:eid AND LENGTH($CL)>0 AND YEAR($CL)=YEAR(CURDATE()) AND MONTH($CL)=MONTH(CURDATE())";
    $st=$pdo->prepare($sql); $st->bindParam(':eid',$eid); $st->execute();
    return (int)$st->fetchColumn();
  } catch (Throwable $e){ return 0; }
}

$total_pl    = kpi_sum_today($pdo,$T,$E,$eid,$PF,$CL,$OP);
$win_rate    = kpi_win_rate_30d($pdo,$T,$E,$eid,$PF,$CL,$OP);
$open_trades = kpi_open_trades_guess($pdo,$T,$E,$eid,$CL);
$closed_trades = kpi_closed_this_month($pdo,$T,$E,$eid,$CL);

// Page helpers
$current_script = basename(parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH));

// Account-type mapping from your trades.php (campaigns)
function map_account_from_campaign($c){
  $c = (string)$c;
  if ($c === 'AURO\MTL\MTL02') return 'RAW ECN Account';
  if ($c === 'AURO\MTL\MTL01') return 'AI Account';
  if ($c === 'AURO\MTL\MTL03') return 'Standard Account';
  // fallback: show original
  return $c !== '' ? $c : '—';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <?php include 'title.php'; ?>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Trades</title>
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css"/>

  <style>
    :root{
      --orange-600:#F1671B; --orange-500:#ff7a00; --orange-50:#FFF5ED;
      --ink-900:#0f172a; --ink-700:#334155; --ink-500:#64748b;
      --border:#e5e7eb; --bg:#ffffff; --card:#ffffff; --muted:#f8fafc;
      --ring: rgba(241,103,27,.25);
      --shadow:0 10px 25px rgba(15,23,42,.06), 0 2px 8px rgba(15,23,42,.05);
    }
    html,body{font-family:'Inter',system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;background:var(--orange-50);color:var(--ink-900)}
    .brand-navbar{background:var(--bg);border-bottom:1px solid var(--border);position:sticky;top:0;z-index:40}
    .brand-navbar .nav-link{font-weight:600;color:var(--ink-700);border-radius:10px;padding:.6rem .9rem!important}
    .brand-navbar .nav-link.active,.brand-navbar .nav-link:hover{color:var(--orange-600);background:#fff;box-shadow:inset 0 0 0 2px var(--orange-50)}
    .brand-badge{background:var(--orange-600);color:#fff;padding:.45rem .75rem;border-radius:10px;font-weight:700;letter-spacing:.3px}

    .top{position:sticky;top:0;z-index:50;backdrop-filter:saturate(180%) blur(8px);background:rgba(255,255,255,.85);border-bottom:1px solid var(--border)}
    .top-inner{max-width:1200px;margin:0 auto;padding:12px 18px;display:flex;gap:12px;align-items:center}
    .logo{display:flex;align-items:center;gap:10px;font-weight:800}
    .logo-b{width:36px;height:36px;border-radius:12px;display:grid;place-items:center;color:#fff;background:linear-gradient(135deg,var(--orange-600),#ff9952);box-shadow:var(--shadow)}
    .grow{flex:1}
    .search{display:flex;align-items:center;gap:8px;background:#fff;border:1px solid var(--border);border-radius:999px;padding:8px 12px}
    .search input{border:0;outline:0;width:100%;font:inherit}
    .btn{appearance:none;border:0;border-radius:999px;padding:10px 16px;font-weight:700;cursor:pointer}
    .btn-primary{background:var(--orange-600)!important;border-color:var(--orange-600)!important;color:#fff}
    .btn-ghost{background:transparent;color:var(--orange-600)}
    .btn-chip{background:#fff7f0;border:1px solid #ffd9bf;color:#b45309;font-weight:700;padding:8px 12px;border-radius:999px;cursor:pointer}
    .btn-chip.active{outline:2px solid #ffd9bf}

    .wrap{max-width:1200px;margin:24px auto;padding:0 18px}
    .head{display:flex;flex-wrap:wrap;align-items:flex-end;justify-content:space-between;gap:12px}
    .title{font-size:28px;font-weight:800;margin:0}
    .crumbs{font-size:12px;color:var(--ink-500)}

    .grid{display:grid;gap:14px}
    .g-4{grid-template-columns:repeat(4,minmax(0,1fr))}
    @media(max-width:1100px){ .g-4{grid-template-columns:repeat(2,minmax(0,1fr))} }
    @media(max-width:640px){ .g-4{grid-template-columns:1fr} }

    .card{border:1px solid var(--border);border-radius:14px;background:#fff;box-shadow:var(--shadow)}
    .card-h{display:flex;align-items:center;justify-content:space-between;padding:16px 16px 0 16px}
    .card-t{font-size:12px;color:var(--ink-500);text-transform:uppercase;letter-spacing:.08em;font-weight:700}
    .badge{display:inline-flex;align-items:center;gap:6px;background:#fff7f0;color:#c2410c;padding:6px 10px;border-radius:999px;font-size:12px;font-weight:700}
    .card-b{padding:16px}
    .kpi{font-size:26px;font-weight:800}
    .kpi-sub{font-size:12px;color:var(--ink-500);margin-top:6px}
    .up{color:#16a34a}.down{color:#dc2626}

    .filters{margin-top:16px;background:#fff;border:1px solid var(--border);border-radius:18px;box-shadow:var(--shadow)}
    .filters .row{display:grid;grid-template-columns:1fr auto;gap:10px;padding:16px}
    @media(max-width:800px){.filters .row{grid-template-columns:1fr}}
    .lab{font-size:12px;color:var(--ink-500);text-transform:uppercase;letter-spacing:.08em;font-weight:700}
    .in{padding:10px 12px;border:1px solid var(--border);border-radius:12px;background:#fff;width:100%}
    .dates{display:flex;gap:8px}
    .chips{display:flex;flex-wrap:wrap;gap:8px}

    .panel{margin-top:16px;background:#fff;border:1px solid var(--border);border-radius:18px;box-shadow:var(--shadow)}
    .panel-h{display:flex;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid var(--border)}
    .panel-t{font-weight:800}
    .table-wrap{overflow:auto}
    table{width:100%;border-collapse:separate;border-spacing:0}
    thead th{position:sticky;top:0;background:#fff;border-bottom:1px solid var(--border);text-align:left;font-size:12px;color:var(--ink-500);text-transform:uppercase;letter-spacing:.08em;padding:14px}
    tbody td{padding:14px;border-bottom:1px solid #faf3ec;vertical-align:middle}
    tbody tr:hover{background:#fffaf5}
    .pill{padding:6px 10px;border-radius:999px;font-size:12px;font-weight:700;display:inline-flex;align-items:center;gap:6px}
    .ok{background:#ecfdf5;color:#16a34a}
    .bad{background:#fef2f2;color:#dc2626}
    .pl-pos{color:#16a34a;font-weight:800}
    .pl-neg{color:#dc2626;font-weight:800}
  </style>
</head>
<body>
  <?php include 'header.php'; ?>

  <nav class="navbar navbar-expand-lg brand-navbar">
    <div class="container-fluid">
      <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav"
              aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation"
              style="border-color: var(--orange-600);">
        <span class="navbar-toggler-icon"></span>
      </button>

      <div class="collapse navbar-collapse" id="navbarNav">
        <ul class="navbar-nav ms-auto gap-2">
          <li class="nav-item"><a class="nav-link" href="dashboard"><i class="fa fa-home me-1"></i>Dashboard</a></li>
          <li class="nav-item"><a class="nav-link" href="clients?level=1"><i class="fa fa-users me-1"></i>Clients</a></li>
          <li class="nav-item"><a class="nav-link active" href="<?= htmlspecialchars($current_script) ?>"><i class="fa fa-bar-chart me-1"></i>Trade List</a></li>
          <li class="nav-item"><a class="nav-link" href="withdraw"><i class="fa fa-institution me-1"></i>Payout</a></li>
        </ul>
      </div>
    </div>
  </nav>

  <header class="top">
    <div class="top-inner">
      <div class="logo"><div class="logo-b">T</div><span>Trade Desk</span></div>
      <div class="grow"></div>
      <button class="btn btn-ghost" onclick="history.back()">Back</button>
      <button class="btn btn-primary" onclick="openDrawer('new-trade')">New Trade</button>
    </div>
  </header>

  <main class="wrap">
    <div class="head">
      <div>
        <div class="crumbs">Dashboard / Trades</div>
        <h1 class="title">Trades</h1>
      </div>
      <div class="muted">As of <?= htmlspecialchars(date('d M Y, h:i A')) ?></div>
    </div>

    <!-- FILTERS -->
    <section class="filters">
      <form class="row" method="get" action="<?= htmlspecialchars($current_script) ?>">
        <div style="display:flex;flex-direction:column;gap:8px">
          <label class="lab" for="searchTable2">Quick Search</label>
          <input class="in" id="searchTable2" type="text" placeholder="Type to filter by any column…">
        </div>

        <div style="display:flex;flex-direction:column;gap:8px;min-width:460px">
          <div class="lab">Datewise Filter (Close Time → Open Time)</div>
          <div class="dates">
            <input class="in" type="date" name="from" id="from" value="<?= htmlspecialchars($from) ?>">
            <input class="in" type="date" name="to"   id="to"   value="<?= htmlspecialchars($to) ?>">
            <button class="btn btn-primary" type="submit">Apply</button>
            <a class="btn btn-ghost" href="<?= htmlspecialchars($current_script) ?>">Reset</a>
          </div>
          <div class="chips">
            <button type="button" class="btn-chip" onclick="presetRange('today')">Today</button>
            <button type="button" class="btn-chip" onclick="presetRange('7')">Last 7 Days</button>
            <button type="button" class="btn-chip" onclick="presetRange('month')">This Month</button>
            <button type="button" class="btn-chip" onclick="presetRange('year')">This Year</button>
          </div>
        </div>

        <div style="display:flex;gap:8px;align-items:flex-end;justify-content:flex-end">
          <button class="btn btn-ghost" type="button" onclick="exportCSV()">Export CSV</button>
        </div>
      </form>
    </section>

    <section class="panel">
      <div class="panel-h">
        <div class="panel-t">
          All Trades<?= ($from && $to) ? ' — '.htmlspecialchars($from).' to '.htmlspecialchars($to) : '' ?>
          <?= ($from && $to) ? '' : '' ?>
        </div>
        <div></div>
      </div>
      <div class="table-wrap">
        <table id="tradesTable">
          <thead>
            <tr>
              <th>Trading Account</th>
              <th>Platform</th>
              <th>Acc Type</th>
              <th>Symbols</th>
              <th>Action</th>
              <th>Volume</th>
              <th>Open Time</th>
              <th>Close Time</th>
              <th>Profit</th>
            </tr>
          </thead>
          <tbody>
            <?php if(!empty($rows)): foreach($rows as $r):
              $acct = map_account_from_campaign($r['campaigns'] ?? '');
              $profit = (float)($r['profit'] ?? 0);
              $plClass = $profit>=0 ? 'pl-pos' : 'pl-neg';
            ?>
              <tr>
                <td><?= htmlspecialchars($r['trade_id'] ?? '') ?></td>
                <td>MT5</td>
                <td><?= htmlspecialchars($acct) ?></td>
                <td><?= htmlspecialchars($r['symbol'] ?? '') ?></td>
                <td><span class="pill <?= strtolower($r['action']??'')==='buy' ? 'ok' : 'bad' ?>"><?= htmlspecialchars(strtoupper($r['action'] ?? '')) ?></span></td>
                <td><?= htmlspecialchars($r['vol'] ?? '') ?></td>
                <td><?= htmlspecialchars($r['open_time'] ?? '') ?></td>
                <td><?= htmlspecialchars($r['close_time'] ?? '') ?></td>
                <td class="<?= $plClass ?>">$ <?= number_format($profit,2) ?></td>
              </tr>
            <?php endforeach; else: ?>
              <tr><td colspan="9">No trades found<?= ($from||$to)?' for selected date(s)':'' ?>.</td></tr>
            <?php endif; ?>
          </tbody>
        </table>
      </div>
    </section>

    <?php include 'footer.php'; ?>
  </main>

  <!-- Drawer (UI helper) -->
  <div id="drawer" style="display:none;position:fixed;inset:0;background:rgba(0,0,0,.35);align-items:flex-end">
    <div id="drawerCard" style="width:100%;max-width:560px;margin-left:auto;background:#fff;border-top-left-radius:24px;border-top-right-radius:24px;box-shadow:0 10px 24px rgba(0,0,0,.06);min-height:40vh">
      <div class="d-h" style="display:flex;align-items:center;justify-content:space-between;padding:18px;border-bottom:1px solid #f1f1f1">
        <strong id="drawerTitle">Action</strong>
        <button class="btn btn-ghost" onclick="closeDrawer()">Close</button>
      </div>
      <div class="d-b" id="drawerBody" style="padding:18px">
        <p class="muted">Paste your existing PHP form or trade details here.</p>
      </div>
    </div>
  </div>

<script>
// Expose PHP dates safely to JS
const DATE_FROM = <?= json_encode($from ?? '') ?>;
const DATE_TO   = <?= json_encode($to ?? '') ?>;

// CSV exporter (header + visible rows only)
function exportCSV(){
  const table = document.getElementById('tradesTable');
  if (!table) return;

  const lines = [];

  // Header
  if (table.tHead && table.tHead.rows.length){
    const headCells = [...table.tHead.rows[0].cells].map(td => csvEscape(td.innerText));
    lines.push(headCells.join(','));
  }

  // Body (skip hidden rows after client-side search filter)
  if (table.tBodies && table.tBodies[0]){
    [...table.tBodies[0].rows].forEach(tr => {
      if (tr.style.display === 'none') return; // respect current filter
      const row = [...tr.cells].map(td => csvEscape(td.innerText));
      lines.push(row.join(','));
    });
  }

  // Build CSV with BOM for Excel
  const csv = '\uFEFF' + lines.join('\n');
  const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });

  // Filename with date range if present
  const range = (DATE_FROM && DATE_TO) ? `-${DATE_FROM}-to-${DATE_TO}` : '';
  const a = document.createElement('a');
  a.href = URL.createObjectURL(blob);
  a.download = `trades${range}.csv`;
  document.body.appendChild(a);
  a.click();
  a.remove();
}

// Proper CSV escaping: double quotes inside quoted field; strip newlines
function csvEscape(text){
  const t = (text || '').replace(/\r?\n|\r/g, ' ').trim().replace(/"/g, '""');
  return `"${t}"`;
}
</script>

  <!-- Keep if your layout expects these -->
  <script src="assets/js/core/libs.min.js"></script>
  <script src="assets/js/core/external.min.js"></script>
  <script src="assets/js/plugins/setting.js"></script>
  <script src="assets/js/hope-ui.js" defer></script>
</body>
</html>
