<?php
// tree_chart.php  (PDO version)
if (session_status() === PHP_SESSION_NONE) { session_start(); }
require_once 'config.php'; // MUST define $conn as PDO

// Show errors in dev (remove on prod)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$rootEmail = isset($_SESSION['clogin']) ? trim($_SESSION['clogin']) : '';
if ($rootEmail === '') { die("Login required: session 'clogin' is empty."); }

/*
  Fetch all users whose ib1..ib10 equals rootEmail, plus quick account stats.
  We'll use ib1 as the immediate parent/upline to construct the tree.
*/
$sql = "
  SELECT
    u.email,
    u.username,
    u.fullname,
    u.number AS phone,
    u.reg_date AS registered_at,
    u.ib1, u.ib2, u.ib3, u.ib4, u.ib5, u.ib6, u.ib7, u.ib8, u.ib9, u.ib10,
    la.mt5_id,
    IFNULL(la.total_lots,0)       AS total_lots,
    IFNULL(la.total_deposit,0.0)  AS total_deposit,
    IFNULL(la.total_withdraw,0.0) AS total_withdraw,
    CASE
      WHEN u.ib1  = ? THEN 1 WHEN u.ib2  = ? THEN 2 WHEN u.ib3  = ? THEN 3
      WHEN u.ib4  = ? THEN 4 WHEN u.ib5  = ? THEN 5 WHEN u.ib6  = ? THEN 6
      WHEN u.ib7  = ? THEN 7 WHEN u.ib8  = ? THEN 8 WHEN u.ib9  = ? THEN 9
      WHEN u.ib10 = ? THEN 10
    END AS level
  FROM aspnetusers u
  LEFT JOIN (
    SELECT
      email,
      SUBSTRING_INDEX(GROUP_CONCAT(trade_id ORDER BY id DESC), ',', 1) AS mt5_id,
      SUM(lotsCompleted)  AS total_lots,
      SUM(deposit)        AS total_deposit,
      SUM(withdraw)       AS total_withdraw
    FROM liveaccount
    GROUP BY email
  ) la ON la.email = u.email
  WHERE (u.ib1  = ? OR u.ib2  = ? OR u.ib3  = ? OR u.ib4  = ? OR u.ib5  = ?
      OR  u.ib6  = ? OR u.ib7  = ? OR u.ib8  = ? OR u.ib9  = ? OR u.ib10 = ?)
  ORDER BY level, u.email
";

// Prepare + execute (20 placeholders, all same email)
$stmt = $conn->prepare($sql);
$binds = array_fill(0, 20, $rootEmail);
$stmt->execute($binds);
$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Build nodes + parent/children using ib1 as direct parent
$nodes = [];
$childrenMap = [];

// Synthesize root (you) in case you don't exist in aspnetusers
$nodes[$rootEmail] = [
  'email'      => $rootEmail,
  'name'       => '(You)',
  'phone'      => '',
  'mt5_id'     => '',
  'lots'       => 0,
  'deposit'    => 0,
  'withdraw'   => 0,
  'registered' => '',
  'level'      => 0
];
$childrenMap[$rootEmail] = [];

foreach ($rows as $r) {
  $email = $r['email'];
  $name  = $r['fullname'] ?: $r['username'] ?: '';

  $nodes[$email] = [
    'email'      => $email,
    'name'       => $name,
    'phone'      => $r['phone'] ?: '',
    'mt5_id'     => $r['mt5_id'] ?: '',
    'lots'       => (float)$r['total_lots'],
    'deposit'    => (float)$r['total_deposit'],
    'withdraw'   => (float)$r['total_withdraw'],
    'registered' => $r['registered_at'] ?: '',
    'level'      => (int)$r['level']
  ];

  $parentEmail = trim($r['ib1'] ?? '');
  if ($parentEmail === '') { $parentEmail = $rootEmail; } // fallback
  if (!isset($childrenMap[$parentEmail])) $childrenMap[$parentEmail] = [];
  $childrenMap[$parentEmail][] = $email;

  if (!isset($childrenMap[$email])) $childrenMap[$email] = [];
}

// Build nested tree (cap at 10 levels)
function buildTree($root, &$nodes, &$childrenMap, $maxDepth = 10, $depth = 0) {
  if (!isset($nodes[$root])) return null;
  if ($depth > $maxDepth) return null;

  $item = $nodes[$root];
  $item['children'] = [];

  if (!empty($childrenMap[$root])) {
    foreach ($childrenMap[$root] as $childEmail) {
      if (isset($nodes[$childEmail]) && $nodes[$childEmail]['level'] >= 1 && $nodes[$childEmail]['level'] <= 10) {
        $childNode = buildTree($childEmail, $nodes, $childrenMap, $maxDepth, $depth + 1);
        if ($childNode) $item['children'][] = $childNode;
      }
    }
  }
  return $item;
}

$tree = buildTree($rootEmail, $nodes, $childrenMap, 10);
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>IB Hierarchy Tree</title>
<meta name="viewport" content="width=device-width, initial-scale=1" />
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<style>
  :root{
    --brand:#ff6a00; --ink:#0f172a; --muted:#64748b; --bg:#f8fafc; --card:#ffffff; --line:#e2e8f0;
  }
  body{background:var(--bg); color:var(--ink);}
  .wrap{max-width:1200px; margin:24px auto; padding:0 16px;}
  .toolbar{display:flex; gap:8px; flex-wrap:wrap; align-items:center; margin-bottom:12px;}
  .tree-wrap{
    background:var(--card); border-radius:16px; box-shadow:0 6px 22px rgba(2,6,23,.06);
    padding:16px; overflow:auto; min-height:60vh;
  }

  /* Vertical tree with connectors */
  .tree{ position: relative; padding-left: 0; }
  .tree ul{ padding-left: 24px; margin: 0; list-style: none; position: relative; }
  .tree li{ position: relative; padding-left: 16px; margin: 10px 0; }
  .tree li::before{
    content:""; position:absolute; left:0; top:1.2em; width:12px; height:0; border-top:1px solid var(--line);
  }
  .tree ul::before{
    content:""; position:absolute; left:6px; top:0; bottom:0; border-left:1px solid var(--line);
  }

  .node{
    background:#fff; border:1px solid #e5e7eb; border-radius:12px; padding:10px 12px;
    display:inline-block; min-width: 260px; box-shadow:0 2px 12px rgba(2,6,23,.04);
  }
  .node .title{ font-weight:700; margin:0; display:flex; align-items:center; gap:8px; }
  .badge-level{ background: var(--brand); color:#fff; border-radius:999px; padding:2px 8px; font-size:11px; font-weight:700; }
  .node .meta{ font-size:12px; color:var(--muted); }
  .node .kv{ font-size:12px; margin-top:4px; display:flex; gap:12px; flex-wrap:wrap; }
  .toggle{
    border:0; background:#eef2ff; color:#1e293b; font-weight:700; border-radius:6px; padding:2px 8px; margin-right:6px; cursor:pointer;
  }
  .collapsed > ul{ display:none; }
  .legend{font-size:12px; color:var(--muted);}
</style>
</head>
<body class="min-h-screen bg-gradient-to-b from-brand-50 to-white text-gray-900 transition-colors duration-300">

<?php include __DIR__ . '/header.php'; ?>
<div class="flex flex-col md:flex-row">
  <?php include __DIR__ . '/side_bar.php'; ?>

  <main class="flex-1 p-4 sm:p-6 md:ml-64">
      <div class="wrap">
  <div class="d-flex justify-content-between align-items-center mb-2">
   <div class="wrap">
  <div class="d-flex justify-content-between align-items-center mb-2">
    <div>
      <h3 class="mb-0">IB Hierarchy</h3>
      <div class="text-muted">Root: <strong><?= htmlspecialchars($rootEmail, ENT_QUOTES, 'UTF-8'); ?></strong></div>
    </div>
    <div class="legend">Tip: Click the chevron to expand/collapse. Search opens matching branches.</div>
  </div>

  <div class="toolbar">
    <input id="search" type="text" class="form-control" style="max-width:320px" placeholder="Search name/email/MT5 ID...">
    <button class="btn btn-outline-secondary btn-sm" id="expandAll">Expand All</button>
    <button class="btn btn-outline-secondary btn-sm" id="collapseAll">Collapse All</button>
  </div>

  <div class="tree-wrap">
    <div id="tree" class="tree"></div>
  </div>
</div>
</div>
</div></main>
</div>
<script>
const DATA = <?= json_encode($tree ?? [], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); ?>;

function escapeHtml(s){ return (s??'').toString().replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[m])); }
function fmtLots(n){ const x = Number(n||0); const s = x.toFixed(2); return s.replace(/\.00$/,'').replace(/(\.\d)0$/,'$1'); }

function nodeHtml(n){
  const name = n.level===0 ? '(You)' : (n.name || '-');
  return `
    <div class="node">
      <p class="title">
        ${n.children && n.children.length ? '<button class="toggle" title="Expand/Collapse">▾</button>' : '<span style="width:28px;display:inline-block;"></span>'}
        <span>${escapeHtml(name)}</span>
        <span class="badge-level">L${n.level||0}</span>
      </p>
      <div class="meta">${escapeHtml(n.email||'')}</div>
      <div class="kv">
        <span><strong>MT5:</strong> ${escapeHtml(n.mt5_id||'-')}</span>
        <span><strong>Lots:</strong> ${fmtLots(n.lots)}</span>
        <span><strong>Dep:</strong> $${Number(n.deposit||0).toLocaleString()}</span>
        <span><strong>Wdr:</strong> $${Number(n.withdraw||0).toLocaleString()}</span>
        <span><strong>Reg:</strong> ${escapeHtml(n.registered||'-')}</span>
        <span><strong>Phone:</strong> ${escapeHtml(n.phone||'-')}</span>
      </div>
    </div>
  `;
}

function buildList(n){
  if (!n || !n.email) {
    const div = document.createElement('div');
    div.innerHTML = '<em>No downline yet.</em>';
    return div;
  }
  const li = document.createElement('li');
  li.innerHTML = nodeHtml(n);
  if (n.children && n.children.length){
    const ul = document.createElement('ul');
    n.children.forEach(ch => ul.appendChild(buildList(ch)));
    li.appendChild(ul);
  }
  return li;
}

function renderTree(){
  const container = document.getElementById('tree');
  container.innerHTML = '';
  const rootUl = document.createElement('ul');
  rootUl.appendChild(buildList(DATA));
  container.appendChild(rootUl);

  // Toggle handlers
  container.querySelectorAll('.toggle').forEach(btn=>{
    btn.addEventListener('click', ()=>{
      const li = btn.closest('li');
      li.classList.toggle('collapsed');
      btn.textContent = li.classList.contains('collapsed') ? '▸' : '▾';
    });
  });
}

function expandAll(){
  document.querySelectorAll('#tree li.collapsed').forEach(li=>{
    li.classList.remove('collapsed');
    const btn = li.querySelector('.toggle'); if (btn) btn.textContent = '▾';
  });
}
function collapseAll(){
  document.querySelectorAll('#tree li').forEach(li=>{
    if (li.querySelector('ul')) {
      li.classList.add('collapsed');
      const btn = li.querySelector('.toggle'); if (btn) btn.textContent = '▸';
    }
  });
  // keep root open
  const root = document.querySelector('#tree > ul > li');
  if (root) { root.classList.remove('collapsed'); const btn=root.querySelector('.toggle'); if (btn) btn.textContent='▾'; }
}

renderTree();

// Search + auto-expand matches
const search = document.getElementById('search');
search.addEventListener('input', ()=>{
  const q = search.value.trim().toLowerCase();
  document.querySelectorAll('#tree li').forEach(li => li.style.outline = '');

  if (!q) return;

  document.querySelectorAll('#tree .node').forEach(card=>{
    const text = card.innerText.toLowerCase();
    if (text.includes(q)) {
      const li = card.closest('li');
      // open all ancestors
      let p = li.parentElement;
      while (p && p.id !== 'tree'){
        if (p.tagName === 'UL') {
          const pli = p.parentElement;
          pli.classList.remove('collapsed');
          const btn = pli.querySelector('.toggle'); if (btn) btn.textContent='▾';
        }
        p = p.parentElement;
      }
      li.style.outline = '2px solid #22c55e';
    }
  });
});

// Buttons
document.getElementById('expandAll').addEventListener('click', expandAll);
document.getElementById('collapseAll').addEventListener('click', collapseAll);
</script>
</body>
</html>
