<?php
session_start();
ini_set('display_errors', 1);
error_reporting(E_ALL);
include "config.php";

if (empty($_SESSION['clogin'])) {
  echo "<script>window.location.href='login.php';</script>";
  exit();
}

$userEmail = $_SESSION['clogin'];

// -------------------------
// Helper: strong password
// -------------------------
function is_strong_pwd($pwd) {
  return (bool) preg_match('/^(?=.*[A-Z])(?=.*\d)(?=.*[^A-Za-z0-9]).{8,}$/', $pwd);
}

// -------------------------
// CSRF Token
// -------------------------
if (empty($_SESSION['csrf_change_pwd'])) {
  $_SESSION['csrf_change_pwd'] = bin2hex(random_bytes(32));
}
$csrf = $_SESSION['csrf_change_pwd'];

// -------------------------
// Load all accounts for user
// -------------------------
$accounts = [];
try {
  $stmt = $conn->prepare("SELECT trade_id, trader_pwd, invester_pwd FROM liveaccount WHERE email = ?");
  $stmt->execute([$userEmail]);
  $accounts = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
  $accounts = [];
}

$flash = ['type'=>null,'msg'=>null];

// -------------------------
// Handle submit
// -------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $posted_csrf   = $_POST['csrf'] ?? '';
  $trade_id      = trim($_POST['tradeid'] ?? '');
  $password_type = $_POST['password_type'] ?? '';
  $current_pwd   = $_POST['current_pwd'] ?? '';
  $new_pwd       = $_POST['new_pwd'] ?? '';
  $confirm_pwd   = $_POST['confirm_pwd'] ?? '';

  if (!hash_equals($csrf, $posted_csrf)) {
    $flash = ['type'=>'error','msg'=>'Invalid request (CSRF). Please refresh and try again.'];
  } elseif (!$trade_id || !in_array($password_type, ['trader','investor'], true)) {
    $flash = ['type'=>'error','msg'=>'Please select MT5 Account ID and Password Type.'];
  } elseif ($new_pwd !== $confirm_pwd) {
    $flash = ['type'=>'error','msg'=>'New Password and Confirm Password do not match.'];
  } elseif (!is_strong_pwd($new_pwd)) {
    $flash = ['type'=>'error','msg'=>'Password must be ≥ 8 chars with 1 uppercase, 1 number, and 1 special character.'];
  } else {
    // Re-fetch the chosen account to verify current password and update
    $stmt = $conn->prepare("SELECT trader_pwd, invester_pwd FROM liveaccount WHERE email = ? AND trade_id = ? LIMIT 1");
    $stmt->execute([$userEmail, $trade_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$row) {
      $flash = ['type'=>'error','msg'=>'Account not found for this user.'];
    } else {
      $stored_current = ($password_type === 'trader') ? ($row['trader_pwd'] ?? '') : ($row['invester_pwd'] ?? '');
      // Optional: enforce verifying current password typed by user
      if ($stored_current !== $current_pwd) {
        $flash = ['type'=>'error','msg'=>'Current password is incorrect.'];
      } else {
        try {
          if ($password_type === 'trader') {
            $upd = $conn->prepare("UPDATE liveaccount SET trader_pwd = ? WHERE email = ? AND trade_id = ? LIMIT 1");
            $upd->execute([$new_pwd, $userEmail, $trade_id]);
          } else {
            $upd = $conn->prepare("UPDATE liveaccount SET invester_pwd = ? WHERE email = ? AND trade_id = ? LIMIT 1");
            $upd->execute([$new_pwd, $userEmail, $trade_id]);
          }

          // TODO: If you integrate MT5 API, call it here and handle failures accordingly.
          // e.g. mt5_change_password($trade_id, $new_pwd, $password_type);

          $flash = ['type'=>'success','msg'=>'Password updated successfully.'];

          // Refresh accounts in memory so UI shows new current password immediately
          $stmt = $conn->prepare("SELECT trade_id, trader_pwd, invester_pwd FROM liveaccount WHERE email = ?");
          $stmt->execute([$userEmail]);
          $accounts = $stmt->fetchAll(PDO::FETCH_ASSOC);

        } catch (Throwable $e) {
          $flash = ['type'=>'error','msg'=>'Update failed. Please try again.'];
        }
      }
    }
  }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title>Change MT5 Password</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css" rel="stylesheet">
  <style>
    .hint { font-size:12px; color:#6b7280; }
    .err  { color:#b91c1c; font-size:12px; margin-top:6px; display:none; }
    .ok   { color:#065f46; font-size:12px; margin-top:6px; display:none; }
    .input-bad { border-color:#ef4444 !important; }
  </style>
</head>
<body class="bg-gray-100">
<?php include "header.php"; ?>
<div class="flex flex-col md:flex-row min-h-screen">
  <?php include "side_bar.php"; ?>

  <main class="flex-1 p-6 md:ml-64" style="width:100%;">
    <div class="container">
      <span class="text-[#CD7F32] font-semibold text-lg">Change MT5 Password</span>
    </div>

    <div class="container mx-auto p-6">
      <h2 class="text-2xl font-bold text-[#CD7F32] mb-4 text-center">Select MT5 Account and Update Password</h2>

      <?php if ($flash['type']==='success'): ?>
        <div class="mb-4 rounded-md bg-green-50 p-4 text-green-700 border border-green-200">
          <?= htmlspecialchars($flash['msg']) ?>
        </div>
      <?php elseif ($flash['type']==='error'): ?>
        <div class="mb-4 rounded-md bg-red-50 p-4 text-red-700 border border-red-200">
          <?= htmlspecialchars($flash['msg']) ?>
        </div>
      <?php endif; ?>

      <?php if (empty($accounts)): ?>
        <div class="bg-white p-6 rounded-lg shadow-md w-full max-w-lg mx-auto">
          <p class="text-gray-700">No MT5 accounts found for your login.</p>
        </div>
      <?php else: ?>
      <form id="pwdForm" method="POST" class="bg-white p-6 rounded-lg shadow-md w-full max-w-lg mx-auto space-y-6" novalidate>
        <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">

        <!-- MT5 Account ID -->
        <div>
          <label for="tradeid" class="block text-[#CD7F32] text-sm font-semibold mb-2">Select MT5 Account ID</label>
          <select id="tradeid" name="tradeid" required
                  class="w-full p-3 border border-gray-400 rounded-md text-sm focus:outline-none focus:ring-2 focus:ring-[#CD7F32]">
            <option value="">-- Select MT5 Account ID --</option>
            <?php
            $postedTrade = $_POST['tradeid'] ?? '';
            foreach ($accounts as $acc):
              $tid = $acc['trade_id'];
            ?>
              <option value="<?= htmlspecialchars($tid) ?>" <?= ($postedTrade===$tid ? 'selected':'') ?>>
                <?= htmlspecialchars($tid) ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>

        <!-- Password Type -->
        <div>
          <label for="password_type" class="block text-[#CD7F32] text-sm font-semibold mb-2">Select Password Type</label>
          <select id="password_type" name="password_type" required
                  class="w-full p-3 border border-gray-400 rounded-md text-sm focus:outline-none focus:ring-2 focus:ring-[#CD7F32]">
            <?php $postedType = $_POST['password_type'] ?? ''; ?>
            <option value="">-- Select Password Type --</option>
            <option value="trader"   <?= ($postedType==='trader'?'selected':'') ?>>Trader Password</option>
            <option value="investor" <?= ($postedType==='investor'?'selected':'') ?>>Investor Password</option>
          </select>
        </div>

        <!-- Current Password (auto-filled) -->
        <div>
          <label for="current_pwd" class="block text-[#CD7F32] text-sm font-semibold mb-2">Current Password</label>
          <div class="relative">
            <input type="password" id="current_pwd" name="current_pwd"
                   class="w-full p-3 pr-12 border border-gray-400 rounded-md text-sm focus:outline-none focus:ring-2 focus:ring-[#CD7F32]"
                   placeholder="Auto-fetched..." value="<?= isset($_POST['current_pwd']) ? htmlspecialchars($_POST['current_pwd']) : '' ?>" required>
            <button type="button" id="toggle_cur" class="absolute right-3 top-1/2 -translate-y-1/2 text-gray-500">
              <i class="fa-regular fa-eye"></i>
            </button>
          </div>
          <div class="hint">Auto-fills after selecting Account ID and Type. Click the eye to view.</div>
        </div>

        <!-- New Password -->
        <div>
          <label for="new_pwd" class="block text-[#CD7F32] text-sm font-semibold mb-2">New Password</label>
          <input type="password" id="new_pwd" name="new_pwd" required
                 class="w-full p-3 border border-gray-400 rounded-md text-sm focus:outline-none focus:ring-2 focus:ring-[#CD7F32]"
                 placeholder="Min 8 chars incl. A-Z, number, special" value="<?= isset($_POST['new_pwd']) ? htmlspecialchars($_POST['new_pwd']) : '' ?>">
          <div id="ruleErr" class="err">Must be ≥ 8 chars with 1 uppercase, 1 number, 1 special character.</div>
          <div id="ruleOk"  class="ok">Looks good ✅</div>
        </div>

        <!-- Confirm Password -->
        <div>
          <label for="confirm_pwd" class="block text-[#CD7F32] text-sm font-semibold mb-2">Confirm New Password</label>
          <input type="password" id="confirm_pwd" name="confirm_pwd" required
                 class="w-full p-3 border border-gray-400 rounded-md text-sm focus:outline-none focus:ring-2 focus:ring-[#CD7F32]"
                 placeholder="Re-enter new password" value="<?= isset($_POST['confirm_pwd']) ? htmlspecialchars($_POST['confirm_pwd']) : '' ?>">
          <div id="matchErr" class="err">Passwords do not match.</div>
        </div>

        <!-- Submit -->
        <button type="submit"
                class="w-full bg-[#CD7F32] text-white py-3 rounded-md font-semibold text-sm hover:bg-[#B76E2B] transition duration-300">
          Update Password
        </button>
      </form>
      <?php endif; ?>
    </div><?php include "footer.php"; ?>
  </main>
  
</div>

<script>
// --- Embed accounts map for instant auto-fill (⚠ exposes passwords to browser) ---
const accountsData = <?php
  $map = [];
  foreach ($accounts as $a) {
    $map[$a['trade_id']] = [
      'trader'   => (string)($a['trader_pwd'] ?? ''),
      'investor' => (string)($a['invester_pwd'] ?? ''),
    ];
  }
  echo json_encode($map, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);
?>;

const tradeSel = document.getElementById('tradeid');
const typeSel  = document.getElementById('password_type');
const curInp   = document.getElementById('current_pwd');

// eye toggles
const tglCur = document.getElementById('toggle_cur');
tglCur?.addEventListener('click', () => {
  if (!curInp.value) return;
  curInp.type = (curInp.type === 'password') ? 'text' : 'password';
  tglCur.innerHTML = (curInp.type === 'password')
    ? '<i class="fa-regular fa-eye"></i>'
    : '<i class="fa-regular fa-eye-slash"></i>';
});

function autofillCurrent() {
  const tid  = tradeSel?.value || '';
  const type = typeSel?.value || '';
  if (tid && type && accountsData[tid]) {
    curInp.value = accountsData[tid][type] || '';
  } else {
    curInp.value = '';
  }
}
tradeSel?.addEventListener('change', autofillCurrent);
typeSel?.addEventListener('change', autofillCurrent);

// prefill if posted values exist
autofillCurrent();

// strength + match validation
const form     = document.getElementById('pwdForm');
const newInp   = document.getElementById('new_pwd');
const confInp  = document.getElementById('confirm_pwd');
const ruleErr  = document.getElementById('ruleErr');
const ruleOk   = document.getElementById('ruleOk');
const matchErr = document.getElementById('matchErr');

const strongRe = /^(?=.*[A-Z])(?=.*\d)(?=.*[^A-Za-z0-9]).{8,}$/;

function validateStrength() {
  if (!newInp.value) {
    ruleErr.style.display='none'; ruleOk.style.display='none';
    newInp.classList.remove('input-bad'); return true;
  }
  const ok = strongRe.test(newInp.value);
  ruleErr.style.display = ok ? 'none' : 'block';
  ruleOk.style.display  = ok ? 'block' : 'none';
  newInp.classList.toggle('input-bad', !ok);
  return ok;
}
function validateMatch() {
  if (!confInp.value) { matchErr.style.display='none'; confInp.classList.remove('input-bad'); return true; }
  const ok = newInp.value === confInp.value;
  matchErr.style.display = ok ? 'none' : 'block';
  confInp.classList.toggle('input-bad', !ok);
  return ok;
}
newInp.addEventListener('input', () => { validateStrength(); validateMatch(); });
confInp.addEventListener('input', validateMatch);
form?.addEventListener('submit', (e) => {
  const ok1 = validateStrength();
  const ok2 = validateMatch();
  if (!ok1 || !ok2) {
    e.preventDefault();
    alert('Please fix password errors before submitting.');
  }
});
</script>
</body>
</html>
