<?php
// client_levels.php (PDO version)
if (session_status() === PHP_SESSION_NONE) { session_start(); }
require_once 'config.php'; // must provide $conn as a PDO instance

// Show errors during dev (remove in prod)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$loggedEmail = isset($_SESSION['clogin']) ? trim($_SESSION['clogin']) : '';
$levelsData = [1=>[],2=>[],3=>[],4=>[],5=>[],6=>[],7=>[],8=>[],9=>[],10=>[]];

if ($loggedEmail !== '') {
    // Build one compact query (same SQL as before)
    $sql = "
        SELECT
            u.email,
            u.username,
            u.fullname,
            u.number AS phone,
            CASE
              WHEN u.ib1  = ? THEN 1 WHEN u.ib2  = ? THEN 2 WHEN u.ib3  = ? THEN 3
              WHEN u.ib4  = ? THEN 4 WHEN u.ib5  = ? THEN 5 WHEN u.ib6  = ? THEN 6
              WHEN u.ib7  = ? THEN 7 WHEN u.ib8  = ? THEN 8 WHEN u.ib9  = ? THEN 9
              WHEN u.ib10 = ? THEN 10
            END AS level,
            la.mt5_id,
            IFNULL(la.total_lots,0)      AS total_lots,
            IFNULL(la.total_deposit,0.0) AS total_deposit,
            IFNULL(la.total_withdraw,0.0)AS total_withdraw,
            u.reg_date AS registered_at
        FROM aspnetusers u
        LEFT JOIN (
            SELECT
                email,
                SUBSTRING_INDEX(GROUP_CONCAT(trade_id ORDER BY id DESC), ',', 1) AS mt5_id,
                SUM(lotsCompleted)  AS total_lots,
                SUM(deposit)        AS total_deposit,
                SUM(withdraw)       AS total_withdraw
            FROM liveaccount
            GROUP BY email
        ) la ON la.email = u.email
        WHERE (u.ib1=? OR u.ib2=? OR u.ib3=? OR u.ib4=? OR u.ib5=?
            OR  u.ib6=? OR u.ib7=? OR u.ib8=? OR u.ib9=? OR u.ib10=?)
        ORDER BY level, u.email
    ";

    // Ensure PDO error mode is exceptions
    if ($conn instanceof PDO) {
        $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    } else {
        die('config.php must provide a PDO instance in $conn for this file.');
    }

    $stmt = $conn->prepare($sql);

    // 10 emails for CASE + 10 for WHERE = 20 params (all same value)
    $binds = array_fill(0, 20, $loggedEmail);
    $stmt->execute($binds);

    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $lvl = (int)$row['level'];
        if ($lvl >= 1 && $lvl <= 10) {
            $levelsData[$lvl][] = [
                'email'      => $row['email'],
                'fullname'   => $row['fullname'] ?: '',
                'username'   => $row['username'] ?: '',
                'phone'      => $row['phone'] ?: '',
                'mt5_id'     => $row['mt5_id'] ?: '',
                'lots'       => $row['total_lots'] ?: 0,
                'deposit'    => $row['total_deposit'] ?: 0,
                'withdraw'   => $row['total_withdraw'] ?: 0,
                'registered' => $row['registered_at'] ?: ''
            ];
        }
    }
    $stmt = null;
}

// Helper: compute totals for a level
function summarizeLevel(array $rows) {
    $sumLots=0; $sumDep=0.0; $sumWdr=0.0;
    foreach ($rows as $r) {
        $sumLots += (float)$r['lots'];
        $sumDep  += (float)$r['deposit'];
        $sumWdr  += (float)$r['withdraw'];
    }
    return [$sumLots, $sumDep, $sumWdr];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Client Levels</title>
<meta name="viewport" content="width=device-width, initial-scale=1" />
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<style>
  :root {
    --brand: #ff6a00;
    --brand-2:#3b82f6;
    --ink:#0f172a;
    --muted:#64748b;
    --bg:#f8fafc;
    --card:#ffffff;
  }
  body{background:var(--bg); color:var(--ink);}
  .wrap{max-width:1200px; margin:32px auto; padding:0 16px;}
  .level-tabs{display:flex; gap:10px; flex-wrap:wrap; margin-bottom:18px;}
  .level-btn{
    border:0; padding:10px 14px; border-radius:999px; background:#e2e8f0; color:#0f172a; font-weight:600;
  }
  .level-btn.active{ background:var(--brand); color:#fff; }
  .metrics{display:grid; grid-template-columns: repeat(4,1fr); gap:12px; margin-bottom:14px;}
  .metric{
    background:var(--card); border-radius:16px; box-shadow:0 6px 24px rgba(2,6,23,0.06);
    padding:16px;
  }
  .metric .label{color:var(--muted); font-size:13px;}
  .metric .value{font-size:22px; font-weight:800;}
  .cardx{
    background:var(--card); border-radius:16px; box-shadow:0 6px 24px rgba(2,6,23,0.06);
    overflow:hidden;
  }
  .table thead th{
    position: sticky; top:0; z-index:1; background:#f1f5f9;
  }
  .table-hover tbody tr:hover{background:#f8fafc;}
  .toolbar{display:flex; gap:10px; align-items:center; padding:12px; border-bottom:1px solid #e2e8f0;}
  .search{max-width:320px;}
  .empty{height:48px;}
  @media(max-width:900px){
    .metrics{grid-template-columns: repeat(2,1fr);}
  }
  @media(max-width:520px){
    .metrics{grid-template-columns: 1fr;}
  }
</style>
</head>
<body class="min-h-screen bg-gradient-to-b from-brand-50 to-white text-gray-900 transition-colors duration-300">

<?php include __DIR__ . '/header.php'; ?>
<div class="flex flex-col md:flex-row">
  <?php include __DIR__ . '/side_bar.php'; ?>

  <main class="flex-1 p-4 sm:p-6 md:ml-64"><div class="wrap">
  <div class="d-flex justify-content-between align-items-center mb-2">
    <div>
      <h3 class="mb-0">IB Programme – Client Levels</h3>
      <!--<div class="text-muted">Logged in as: <strong><?= htmlspecialchars($loggedEmail ?: '-', ENT_QUOTES, 'UTF-8'); ?></strong></div>-->
    </div>
  </div>

  <!-- Level tabs -->
  <div class="level-tabs" id="levelTabs">
    <?php for ($i=1; $i<=10; $i++): ?>
      <button class="level-btn<?= $i===1?' active':''; ?>" data-level="<?= $i; ?>">
        Client Level <?= $i; ?>
        <span class="badge bg-light text-dark ms-1"><?= count($levelsData[$i]); ?></span>
      </button>
    <?php endfor; ?>
  </div>

  <!-- Metrics -->
  <?php
    [$lots1,$dep1,$wdr1] = summarizeLevel($levelsData[1]);
  ?>
  <div class="metrics" id="metrics">
    <div class="metric">
      <div class="label">Clients</div>
      <div class="value" id="mClients"><?= count($levelsData[1]); ?></div>
    </div>
    <div class="metric">
      <div class="label">Total Lots</div>
      <div class="value" id="mLots"><?= rtrim(rtrim(number_format($lots1,2,'.',''), '0'), '.'); ?></div>
    </div>
    <div class="metric">
      <div class="label">Deposit</div>
      <div class="value" id="mDep">$<?= number_format($dep1, 0); ?></div>
    </div>
    <div class="metric">
      <div class="label">Withdraw</div>
      <div class="value" id="mWdr">$<?= number_format($wdr1, 0); ?></div>
    </div>
  </div>

  <!-- Table card -->
  <div class="cardx">
    <div class="toolbar">
      <input type="text" class="form-control search" id="search" placeholder="Search name / email / MT5 ID...">
      <button class="btn btn-outline-secondary btn-sm" id="exportCsv">Export CSV</button>
    </div>
    <div class="table-responsive">
      <table class="table table-hover align-middle mb-0" id="dataTable">
        <thead>
          <tr>
           
            
            <th>Email</th>
           
            <th>MT5 ID</th>
            <th>Total Lots</th>
            <th>Deposit</th>
            <th>Withdraw</th>
            <th>Registered</th>
          </tr>
        </thead>
        <tbody id="tbody">
          <?php
          $rows = $levelsData[1];
          if (empty($rows)): ?>
            <tr class="empty"><td colspan="9"></td></tr>
          <?php else:
            $i=1;
            foreach ($rows as $r): ?>
              <tr>
               
                <td><?= htmlspecialchars($r['email'], ENT_QUOTES, 'UTF-8'); ?></td>
             
                <td><?= htmlspecialchars($r['mt5_id'] ?: '-', ENT_QUOTES, 'UTF-8'); ?></td>
                <td><?= rtrim(rtrim(number_format((float)$r['lots'],2,'.',''), '0'), '.'); ?></td>
                <td>$<?= number_format((float)$r['deposit'], 0); ?></td>
                <td>$<?= number_format((float)$r['withdraw'], 0); ?></td>
                <td><?= htmlspecialchars($r['registered'] ?: '-', ENT_QUOTES, 'UTF-8'); ?></td>
              </tr>
            <?php endforeach;
          endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<script>
// In-page dataset for all levels (so tab switching is instant)
const LEVELS = <?= json_encode($levelsData, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE); ?>;

function formatNum(n, decimals=0) {
  const x = Number(n||0);
  return x.toLocaleString(undefined, {maximumFractionDigits:decimals});
}
function formatLots(n){
  const x = Number(n||0);
  const s = x.toFixed(2);
  return s.replace(/\.00$/,'').replace(/(\.\d)0$/,'$1');
}

function renderLevel(level){
  const rows = LEVELS[level] || [];
  const tbody = document.getElementById('tbody');
  tbody.innerHTML = '';

  if (!rows.length) {
    const tr = document.createElement('tr');
    tr.className = 'empty';
    tr.innerHTML = `<td colspan="9"></td>`;
    tbody.appendChild(tr);
  } else {
    let i=1;
    rows.forEach(r => {
      const tr = document.createElement('tr');
      tr.innerHTML = `
        <td>${escapeHtml(r.email || '-')}</td>
        <td>${escapeHtml(r.mt5_id || '-')}</td>
        <td>${formatLots(Number(r.lots||0))}</td>
        <td>$${formatNum(Number(r.deposit||0))}</td>
        <td>$${formatNum(Number(r.withdraw||0))}</td>
        <td>${escapeHtml(r.registered || '-')}</td>
      `;
      tbody.appendChild(tr);
    });
  }

  // Update metrics
  let clients = rows.length, lots=0, dep=0, wdr=0;
  rows.forEach(r => { lots += Number(r.lots||0); dep += Number(r.deposit||0); wdr += Number(r.withdraw||0); });

  document.getElementById('mClients').textContent = clients;
  document.getElementById('mLots').textContent = formatLots(lots);
  document.getElementById('mDep').textContent = '$' + formatNum(dep);
  document.getElementById('mWdr').textContent = '$' + formatNum(wdr);

  // reset search
  document.getElementById('search').value='';
}

function escapeHtml(s){
  return (s??'').toString().replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[m]));
}

// Tabs
document.querySelectorAll('.level-btn').forEach(btn=>{
  btn.addEventListener('click', ()=>{
    document.querySelectorAll('.level-btn').forEach(b=>b.classList.remove('active'));
    btn.classList.add('active');
    const lvl = Number(btn.dataset.level);
    renderLevel(lvl);
  });
});

// Search (client-side filter on current table)
document.getElementById('search').addEventListener('input', function(){
  const q = this.value.trim().toLowerCase();
  const rows = document.querySelectorAll('#tbody tr');
  rows.forEach(tr=>{
    if (tr.classList.contains('empty')) return;
    const text = tr.innerText.toLowerCase();
    tr.style.display = text.includes(q) ? '' : 'none';
  });
});

// Export CSV of current table
document.getElementById('exportCsv').addEventListener('click', ()=>{
  const rows = [...document.querySelectorAll('#dataTable tr')];
  const csv = rows.map(tr => {
    return [...tr.querySelectorAll('th,td')]
      .map(td => `"${(td.innerText||'').replace(/"/g,'""')}"`).join(',');
  }).join('\n');
  const blob = new Blob([csv], {type:'text/csv;charset=utf-8;'});
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url; a.download = 'clients_level.csv';
  document.body.appendChild(a); a.click(); a.remove();
  URL.revokeObjectURL(url);
});

// Initial render (level 1)
renderLevel(1);
</script>
</body>
</html>
