<?php
/*********************************************************
 * trades.php — All accounts by default; filter when chosen
 *  1) Pull all user's trade_ids from liveaccount
 *  2) If no tradeid specified => merge all accounts
 *  3) MT5: Open Positions + Closed Deals (merged)
 *  4) CSV export + Universal client-side search + Pagination
 *********************************************************/
declare(strict_types=1);
session_start();

// Your original settings
error_reporting(0);
ini_set('display_errors','1');

require_once __DIR__ . '/config.php';             // defines $conn or $dbh (PDO)
require_once __DIR__ . '/manager_initialize.php'; // defines $api, MTRetCode

// ---------- Normalize PDO ----------
$pdo = null;
if (isset($conn) && $conn instanceof PDO) { $pdo = $conn; }
elseif (isset($dbh) && $dbh instanceof PDO) { $pdo = $dbh; }
if (!$pdo) { die('DB not connected — ensure config.php creates $conn or $dbh as PDO.'); }
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

// ---------- Auth guard ----------
if (empty($_SESSION['clogin'])) { header('location:login.php'); exit; }
define('APP_AUTH_OK', true);
$loggedEmail = (string)$_SESSION['clogin'];

// ---------- Helpers ----------
function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
function num($v, int $d=2){ return number_format((float)$v, $d, '.', ','); }

// ---------- 1) Fetch user's accounts ----------
$sql = "
  SELECT id, trade_id, currency, server, account_type, account_name, status, tradePlatform
  FROM liveaccount
  WHERE email = :em
  ORDER BY id DESC
";
$stmt = $pdo->prepare($sql);
$stmt->execute([':em'=>$loggedEmail]);
$accounts = $stmt->fetchAll(PDO::FETCH_ASSOC);
if (!$accounts) { die('No MT accounts found for your login.'); }

$acctByTid = [];
foreach ($accounts as $a) {
  $tid = trim((string)($a['trade_id'] ?? ''));
  if ($tid !== '') $acctByTid[$tid] = $a;
}

// ---------- 2) Input filters ----------
$reqTradeId = isset($_GET['tradeid']) ? trim((string)$_GET['tradeid']) :
              (isset($_POST['tradeid']) ? trim((string)$_POST['tradeid']) : '');

$selectedAll = ($reqTradeId === '' || !isset($acctByTid[$reqTradeId]));

// ---------- 3) Date range (Deals history) ----------
$from = $_GET['from'] ?? $_POST['from'] ?? '';
$to   = $_GET['to']   ?? $_POST['to']   ?? '';
try {
  if ($from === '') $from = (new DateTime('-30 days'))->format('Y-m-d');
  if ($to   === '') $to   = (new DateTime('today'))->format('Y-m-d');
  $fromDT = new DateTime($from . ' 00:00:00');
  $toDT   = new DateTime($to   . ' 23:59:59');
  $fromSec = $fromDT->getTimestamp();
  $toSec   = $toDT->getTimestamp();
} catch (Throwable $e) {
  die('Invalid date range.');
}

// ---------- 4) MT5 fetch helpers ----------
$rowsOpen   = []; // items: ['login'=>int, 'currency'=>string, 'row'=>object]
$rowsDeals  = []; // items: ['login'=>int, 'currency'=>string, 'row'=>object]

// Totals grouped by currency (so mixed accounts don’t mix currencies)
$sumOpenLotsByC   = []; // [currency => lots]
$sumClosedLotsByC = []; // [currency => lots]
$sumClosedPLByC   = []; // [currency => pl]

$pullForLogin = function(int $login, string $fallbackCurrency) use ($api, $fromSec, $toSec, &$rowsOpen, &$rowsDeals, &$sumOpenLotsByC, &$sumClosedLotsByC, &$sumClosedPLByC) {
  // Try to get accurate currency from MT5 account snapshot
  $accObj = null; $cur = $fallbackCurrency ?: 'USD';
  $retAcc = $api->UserAccountGet($login, $accObj);
  if ($retAcc == MTRetCode::MT_RET_OK && !empty($accObj->Currency)) {
    $cur = (string)$accObj->Currency;
  }

  // Open positions
  $totalPositions = 0;
  $ret = $api->PositionGetTotal($login, $totalPositions);
  if ($ret == MTRetCode::MT_RET_OK && (int)$totalPositions > 0) {
    $positions = [];
    $ret = $api->PositionGetPage($login, 0, (int)$totalPositions, $positions);
    if ($ret == MTRetCode::MT_RET_OK && $positions) {
      foreach ($positions as $p) {
        // MT5 volumes are usually in 1/10000 lots → convert to lots
        $lots = (float)($p->Volume ?? 0) * 0.0001;
        $sumOpenLotsByC[$cur] = ($sumOpenLotsByC[$cur] ?? 0) + $lots;
        $rowsOpen[]  = ['login'=>$login, 'currency'=>$cur, 'row'=>$p];
      }
    }
  }

  // Deals history
  $totalDeals = 0;
  $ret = $api->DealGetTotal($login, $fromSec, $toSec, $totalDeals);
  if ($ret == MTRetCode::MT_RET_OK && (int)$totalDeals > 0) {
    $deals = [];
    $ret = $api->DealGetPage($login, $fromSec, $toSec, 0, (int)$totalDeals, $deals);
    if ($ret == MTRetCode::MT_RET_OK && $deals) {
      foreach ($deals as $d) {
        $lots = (float)($d->Volume ?? 0) * 0.0001;
        $pl   = (float)($d->Profit ?? 0);
        $sumClosedLotsByC[$cur] = ($sumClosedLotsByC[$cur] ?? 0) + $lots;
        $sumClosedPLByC[$cur]   = ($sumClosedPLByC[$cur]   ?? 0) + $pl;
        $rowsDeals[] = ['login'=>$login, 'currency'=>$cur, 'row'=>$d];
      }
    }
  }
};

// Pull one or all
if ($selectedAll) {
  foreach ($accounts as $a) {
    $tid = (int)($a['trade_id'] ?? 0); if (!$tid) continue;
    $fallback = (string)($a['currency'] ?? 'USD');
    $pullForLogin($tid, $fallback);
  }
} else {
  $login = (int)$reqTradeId;
  $row   = $acctByTid[$reqTradeId];
  $fallback = (string)($row['currency'] ?? 'USD');
  $pullForLogin($login, $fallback);
}

// Sort by times for readability
usort($rowsOpen, function($A,$B){
  $a = (int)($A['row']->TimeCreate ?? $A['row']->Time ?? 0);
  $b = (int)($B['row']->TimeCreate ?? $B['row']->Time ?? 0);
  return $b <=> $a;
});
usort($rowsDeals, function($A,$B){
  $a = (int)($A['row']->TimeDone ?? $A['row']->TimeExit ?? $A['row']->Time ?? 0);
  $b = (int)($B['row']->TimeDone ?? $B['row']->TimeExit ?? $B['row']->Time ?? 0);
  return $b <=> $a;
});

// Build a short chip string for totals-by-currency
$chipTotals = function(array $m, string $label, int $dec = 2){
  if (!$m) return "$label: 0";
  $parts = [];
  foreach ($m as $cur=>$val) {
    $parts[] = $label . ': ' . num($val, $dec) . ' ' . h($cur);
  }
  return implode(' | ', $parts);
};

?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8" />
  <title>Trades — MT5 (<?= $selectedAll ? 'All Accounts' : h($reqTradeId) ?>)</title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet" />
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet" />
  <style>
    :root { --brand:#ff8c00; --muted:#6c757d; }
    body{ background:#f6f7fb; }
    .page-title{ font-weight:700; }
    .chip{ display:inline-block; padding:.35rem .6rem; border-radius:999px; background:#fff; border:1px solid #eee; font-size:.85rem; }
    .card{ border:0; box-shadow:0 8px 28px rgba(0,0,0,.06); border-radius:16px; }
    .card-header{ background:linear-gradient(90deg, var(--brand), #ffa94d); color:#fff; border-top-left-radius:16px; border-top-right-radius:16px; }
    .table thead th{ background:#fff; position:sticky; top:0; z-index:1; }
    .table-responsive{ max-height:60vh; }
    .toolbar .btn{ border-radius:999px; }
    .muted{ color:var(--muted); }
    .form-select, .form-control { border-radius:12px; }
    .search-input{max-width:320px}

    /* Pagination */
    .pager {
      display:flex; flex-wrap:wrap; gap:.5rem; align-items:center; justify-content:flex-end;
      padding: .75rem 1rem; border-top:1px solid #eee; background:#fff; border-bottom-left-radius:16px; border-bottom-right-radius:16px;
    }
    .pager .info{ margin-right:auto; color:#495057; font-size:.9rem; }
    .pager .btn { border-radius:10px; }
    .pager .pages { display:flex; gap:.25rem; flex-wrap:wrap; }
    .pager .page-btn { min-width:40px; }
    .page-size { width:auto; }
  </style>
  <?php include 'title.php'; ?>
</head>
<body>
<?php include 'header.php'; ?>

<nav class="navbar navbar-expand-lg brand-navbar">
  <div class="container-fluid">
    <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav"
            aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation"
            style="border-color: var(--orange-600);">
      <span class="navbar-toggler-icon"></span>
    </button>

    <div class="collapse navbar-collapse" id="navbarNav">
      <ul class="navbar-nav ms-auto gap-2">
        <li class="nav-item"><a class="nav-link" href="dashboard"><i class="fa fa-home me-1"></i>Dashboard</a></li>
        <li class="nav-item"><a class="nav-link " href=""><i class="fa fa-users me-1"></i>Clients</a></li>
        <li class="nav-item"><a class="nav-link active" href="trades"><i class="fa fa-bar-chart me-1"></i>Trade List</a></li>
        <li class="nav-item"><a class="nav-link" href="withdraw"><i class="fa fa-institution me-1"></i>Payout</a></li>
      </ul>
    </div>
  </div>
</nav>

<div class="container-fluid py-3">

  <!-- Header + chips -->
  <div class="d-flex flex-wrap gap-2 align-items-center justify-content-between mb-3">
    <h1 class="page-title h4 mb-0">Trades</h1>
    <div class="d-flex flex-wrap gap-2">
      <span class="chip">Accounts: <?= $selectedAll ? 'All' : h($reqTradeId) ?></span>
      <span class="chip"><?= h($chipTotals($sumOpenLotsByC, 'Open Vol (lots)', 2)) ?></span>
      <span class="chip"><?= h($chipTotals($sumClosedLotsByC, 'Closed Vol (lots)', 2)) ?></span>
      <span class="chip">
        <?php
          if ($sumClosedPLByC) {
            $parts=[]; foreach ($sumClosedPLByC as $c=>$pl) $parts[]='Closed P/L: '.num($pl,2).' '.h($c);
            echo h(implode(' | ', $parts));
          } else { echo 'Closed P/L: 0'; }
        ?>
      </span>
    </div>
  </div>

  <!-- Account + Date + Global Search -->
  <form class="row gy-2 gx-2 align-items-end mb-3" method="get" id="filtersForm">
    <div class="col-12 col-md-4">
      <label class="form-label mb-0">Account (trade_id)</label>
      <select name="tradeid" class="form-select" onchange="this.form.submit()">
        <option value="" <?= $selectedAll?'selected':'' ?>>All My Accounts</option>
        <?php foreach ($accounts as $a): ?>
          <?php
            $tid = (string)($a['trade_id'] ?? '');
            if ($tid === '') continue;
            $nm = trim((string)($a['account_name'] ?? ''));
            $cr = trim((string)($a['currency'] ?? ''));
            $label = trim($nm . ' ' . $tid . ' ' . $cr);
            if ($label === '') $label = $tid;
          ?>
          <option value="<?= h($tid) ?>" <?= (!$selectedAll && $tid===$reqTradeId)?'selected':'' ?>>
            <?= h($label) ?><?= ($a['status']==='active'?'':' (inactive)') ?>
          </option>
        <?php endforeach; ?>
      </select>
    </div>
    <div class="col-auto">
      <label class="form-label mb-0">From</label>
      <input type="date" class="form-control" name="from" value="<?= h($from) ?>">
    </div>
    <div class="col-auto">
      <label class="form-label mb-0">To</label>
      <input type="date" class="form-control" name="to" value="<?= h($to) ?>">
    </div>
    <div class="col-auto">
      <button class="btn btn-warning text-dark">Apply</button>
      <a class="btn btn-outline-secondary" href="?">Reset</a>
    </div>

    <div class="col-12 col-md ms-md-auto text-md-end">
      <label class="form-label mb-0">Search</label>
      <input type="text" id="globalSearch" class="form-control search-input" placeholder="Type to filter tables...">
    </div>
    <div class="col-auto">
      <button type="button" class="btn btn-outline-primary" onclick="exportCSV()">Export CSV</button>
    </div>
  </form>

  <!-- Open Positions -->
  <div class="card mb-4">
    <div class="card-header">
      <div class="d-flex justify-content-between align-items-center">
        <strong>Open Positions (<?= count($rowsOpen) ?>)</strong>
        <small class="opacity-75"><?= $selectedAll ? 'All accounts' : 'Account '.h($reqTradeId) ?></small>
      </div>
    </div>
    <div class="card-body p-0">
      <div class="table-responsive">
        <table class="table table-sm align-middle mb-0" id="tblOpen">
          <thead>
          <tr>
            <th>#</th>
            <th>Account</th>
            <th>Ticket</th>
            <th>Symbol</th>
            <th>Type</th>
            <th>Volume (lots)</th>
            <th>Open Time</th>
            <th>Open Price</th>
            <th>SL</th>
            <th>TP</th>
            <th>Swap</th>
            <th>Profit</th>
          </tr>
          </thead>
          <tbody>
          <?php if (!$rowsOpen): ?>
            <tr class="no-data"><td colspan="12" class="text-center text-muted py-4">No open positions</td></tr>
          <?php else: $i=1; foreach ($rowsOpen as $w):
            $p   = $w['row']; $acc = $w['login']; $cur = $w['currency'];
            $ticket = $p->Position ?? $p->Ticket ?? '';
            $sym    = $p->Symbol ?? '';
            $type   = $p->Action ?? $p->Type ?? '';
            $typeTxt = is_numeric($type) ? ((int)$type===0?'BUY':((int)$type===1?'SELL':(string)$type)) : (string)$type;
            $volLots= (float)($p->Volume ?? 0) * 0.0001;
            $openTs = (int)($p->TimeCreate ?? $p->Time ?? 0);
            $openAt = $openTs ? date('Y-m-d H:i:s',$openTs) : '';
            $openPx = $p->PriceOpen ?? $p->Price ?? '';
            $sl     = $p->PriceSL ?? '';
            $tp     = $p->PriceTP ?? '';
            $swap   = (float)($p->Storage ?? $p->Swap ?? 0);
            $pl     = (float)($p->Profit ?? 0);
          ?>
            <tr>
              <td data-col="#"><?= $i++ ?></td>
              <td data-col="Account"><?= h($acc) ?></td>
              <td data-col="Ticket"><?= h($ticket) ?></td>
              <td data-col="Symbol"><?= h($sym) ?></td>
              <td data-col="Type"><?= h($typeTxt) ?></td>
              <td data-col="Volume (lots)"><?= num($volLots,2) ?></td>
              <td data-col="Open Time"><?= h($openAt) ?></td>
              <td data-col="Open Price"><?= h($openPx) ?></td>
              <td data-col="SL"><?= h($sl) ?></td>
              <td data-col="TP"><?= h($tp) ?></td>
              <td data-col="Swap"><?= num($swap,2) ?></td>
              <td data-col="Profit" class="<?= $pl>=0?'text-success':'text-danger' ?>"><?= h($cur) ?> <?= num($pl,2) ?></td>
            </tr>
          <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>
      <!-- Pagination controls for Open -->
      <div class="pager" id="pagerOpen">
        <div class="info" id="infoOpen">0–0 of 0</div>
        <div class="d-flex align-items-center gap-2">
          <label class="form-label m-0">Rows:</label>
          <select class="form-select form-select-sm page-size" id="pageSizeOpen">
            <option>10</option><option selected>25</option><option>50</option><option>100</option>
          </select>
        </div>
        <div class="pages" id="pagesOpen"></div>
        <button class="btn btn-outline-secondary btn-sm" id="prevOpen" type="button">&laquo; Prev</button>
        <button class="btn btn-outline-secondary btn-sm" id="nextOpen" type="button">Next &raquo;</button>
      </div>
    </div>
  </div>

  <!-- Closed Deals -->
  <div class="card">
    <div class="card-header">
      <div class="d-flex justify-content-between align-items-center">
        <strong>Closed Deals (<?= count($rowsDeals) ?>) — <?= h($from) ?> to <?= h($to) ?></strong>
        <small class="opacity-75"><?= $selectedAll ? 'All accounts' : 'Account '.h($reqTradeId) ?></small>
      </div>
    </div>
    <div class="card-body p-0">
      <div class="table-responsive">
        <table class="table table-sm align-middle mb-0" id="tblDeals">
          <thead>
          <tr>
            <th>#</th>
            <th>Account</th>
            <th>Deal</th>
            <th>Order</th>
            <th>Symbol</th>
            <th>Type</th>
            <th>Volume (lots)</th>
            <th>Open Price</th>
            <th>Close Price</th>
            <th>Swap</th>
            <th>Profit</th>
          </tr>
          </thead>
          <tbody>
          <?php if (!$rowsDeals): ?>
            <tr class="no-data"><td colspan="11" class="text-center text-muted py-4">No deals in selected range</td></tr>
          <?php else: $i=1; foreach ($rowsDeals as $w):
            $d   = $w['row']; $acc = $w['login']; $cur = $w['currency'];
            $deal    = $d->Deal ?? $d->Ticket ?? '';
            $order   = $d->Order ?? '';
            $sym     = $d->Symbol ?? '';
            $type    = $d->Action ?? $d->Type ?? '';
            $typeTxt = is_numeric($type) ? ((int)$type===0?'BUY':((int)$type===1?'SELL':(string)$type)) : (string)$type;
            $volLots = (float)($d->Volume ?? 0) * 0.0001;
            $op      = $d->PriceOpen  ?? $d->Price ?? '';
            $cp      = $d->PriceClose ?? $d->Price ?? '';
            $swap    = (float)($d->Storage ?? $d->Swap ?? 0);
            $pl      = (float)($d->Profit  ?? 0);
          ?>
            <tr>
              <td data-col="#"><?= $i++ ?></td>
              <td data-col="Account"><?= h($acc) ?></td>
              <td data-col="Deal"><?= h($deal) ?></td>
              <td data-col="Order"><?= h($order) ?></td>
              <td data-col="Symbol"><?= h($sym) ?></td>
              <td data-col="Type"><?= h($typeTxt) ?></td>
              <td data-col="Volume (lots)"><?= num($volLots,2) ?></td>
              <td data-col="Open Price"><?= h($op) ?></td>
              <td data-col="Close Price"><?= h($cp) ?></td>
              <td data-col="Swap"><?= num($swap,2) ?></td>
              <td data-col="Profit" class="<?= $pl>=0?'text-success':'text-danger' ?>"><?= h($cur) ?> <?= num($pl,2) ?></td>
            </tr>
          <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>
      <!-- Pagination controls for Deals -->
      <div class="pager" id="pagerDeals">
        <div class="info" id="infoDeals">0–0 of 0</div>
        <div class="d-flex align-items-center gap-2">
          <label class="form-label m-0">Rows:</label>
          <select class="form-select form-select-sm page-size" id="pageSizeDeals">
            <option>10</option><option selected>25</option><option>50</option><option>100</option>
          </select>
        </div>
        <div class="pages" id="pagesDeals"></div>
        <button class="btn btn-outline-secondary btn-sm" id="prevDeals" type="button">&laquo; Prev</button>
        <button class="btn btn-outline-secondary btn-sm" id="nextDeals" type="button">Next &raquo;</button>
      </div>
    </div>
  </div>

</div>

<script>
// Universal client-side search for both tables
const q = document.getElementById('globalSearch');
q.addEventListener('input', () => {
  const needle = q.value.trim().toLowerCase();
  ['tblOpen','tblDeals'].forEach(id => {
    const table = document.getElementById(id);
    if (!table) return;
    const rows = table.tBodies[0]?.rows || [];
    let anyVisible = false;
    for (const tr of rows) {
      if (tr.classList.contains('no-data')) continue;
      const txt = tr.innerText.toLowerCase();
      const show = (!needle || txt.includes(needle));
      tr.dataset.visible = show ? '1' : '0';
      anyVisible = anyVisible || show;
    }
    // refresh pagination views
    refreshPager(id);
  });
});

// Export both tables to one CSV (filtered rows across all pages)
function exportCSV(){
  const blocks = [
    { id:'tblOpen',  title:'OpenPositions' },
    { id:'tblDeals', title:'ClosedDeals' }
  ];
  let out = [];
  blocks.forEach(b=>{
    const table = document.getElementById(b.id);
    if (!table) return;
    out.push(b.title);
    // header
    const headCells = [...table.tHead.rows[0].children].map(c => csvEscape(c.innerText));
    out.push(headCells.join(','));
    // body: only rows currently marked visible (by search filter); ignore "no-data"
    const rows = table.tBodies[0]?.rows || [];
    for (const r of rows) {
      if (r.classList.contains('no-data')) continue;
      if (r.dataset.visible === '0') continue;
      const cells = [...r.children].map(c => csvEscape(c.innerText));
      out.push(cells.join(','));
    }
    out.push('');
  });
  const a = document.createElement('a');
  a.href = URL.createObjectURL(new Blob([out.join('\n')],{type:'text/csv'}));
  a.download = `trades_<?= $selectedAll ? 'ALL' : h($reqTradeId) ?>_<?= h($from) ?>_to_<?= h($to) ?>.csv`;
  document.body.appendChild(a); a.click(); a.remove();
}
function csvEscape(t){
  const s = (t||'').replace(/\r?\n|\r/g,' ').trim().replace(/"/g,'""');
  return `"${s}"`;
}

/* --------------------------
   Simple Table Pagination
   -------------------------- */
class TablePager {
  constructor(opts){
    this.table     = document.getElementById(opts.tableId);
    this.infoEl    = document.getElementById(opts.infoId);
    this.pagesEl   = document.getElementById(opts.pagesId);
    this.prevBtn   = document.getElementById(opts.prevBtnId);
    this.nextBtn   = document.getElementById(opts.nextBtnId);
    this.sizeSel   = document.getElementById(opts.sizeSelId);
    this.page      = 1;
    this.size      = parseInt(this.sizeSel.value,10) || 25;

    // Mark all data rows visible initially
    this.initVisibility();

    this.sizeSel.addEventListener('change', () => {
      this.size = parseInt(this.sizeSel.value,10) || 25;
      this.page = 1;
      this.render();
    });
    this.prevBtn.addEventListener('click', () => {
      if (this.page > 1) { this.page--; this.render(); }
    });
    this.nextBtn.addEventListener('click', () => {
      if (this.page < this.totalPages()) { this.page++; this.render(); }
    });
    this.render();
  }
  initVisibility(){
    const rows = this.rows();
    for (const r of rows) {
      if (r.classList.contains('no-data')) continue;
      if (!('visible' in r.dataset)) r.dataset.visible = '1';
    }
  }
  rows(){
    return [...(this.table.tBodies[0]?.rows || [])].filter(r => !r.classList.contains('no-data'));
  }
  visibleRows(){
    return this.rows().filter(r => r.dataset.visible !== '0');
  }
  totalPages(){
    const v = this.visibleRows().length;
    return Math.max(1, Math.ceil(v / this.size));
  }
  render(){
    const vrows = this.visibleRows();
    const total = vrows.length;
    const pages = this.totalPages();
    if (this.page > pages) this.page = pages;

    // hide all then show current page slice
    this.rows().forEach(r => r.style.display = 'none');

    const start = (this.page - 1) * this.size;
    const end   = Math.min(start + this.size, total);
    vrows.slice(start, end).forEach(r => r.style.display = '');

    // renumber the "#" column for currently shown slice
    const firstColIsIndex = true;
    if (firstColIsIndex && vrows.length) {
      let i = start + 1;
      vrows.slice(start, end).forEach(r => {
        const cell = r.querySelector('td');
        if (cell) cell.textContent = i++;
      });
    }

    // info text
    const infoTxt = total ? `${start+1}–${end} of ${total}` : `0–0 of 0`;
    this.infoEl.textContent = infoTxt;

    // page buttons
    this.pagesEl.innerHTML = '';
    const MAX_BTN = 7; // compact
    const makeBtn = (p) => {
      const b = document.createElement('button');
      b.type = 'button';
      b.className = 'btn btn-sm ' + (p===this.page ? 'btn-primary' : 'btn-outline-secondary') + ' page-btn';
      b.textContent = p;
      b.addEventListener('click', () => { this.page = p; this.render(); });
      return b;
    };
    const addEllipsis = () => {
      const s = document.createElement('span');
      s.className = 'px-2 text-muted';
      s.textContent = '…';
      this.pagesEl.appendChild(s);
    };

    if (pages <= MAX_BTN) {
      for (let p=1; p<=pages; p++) this.pagesEl.appendChild(makeBtn(p));
    } else {
      // Always show first, last, and a window around current
      const windowSize = 2;
      this.pagesEl.appendChild(makeBtn(1));
      if (this.page > 1 + windowSize + 1) addEllipsis();

      const startP = Math.max(2, this.page - windowSize);
      const endP   = Math.min(pages-1, this.page + windowSize);
      for (let p=startP; p<=endP; p++) this.pagesEl.appendChild(makeBtn(p));

      if (this.page < pages - windowSize - 1) addEllipsis();
      this.pagesEl.appendChild(makeBtn(pages));
    }

    this.prevBtn.disabled = (this.page <= 1);
    this.nextBtn.disabled = (this.page >= pages);
  }
}

// Create pagers
const pagerOpen  = new TablePager({
  tableId:'tblOpen', infoId:'infoOpen', pagesId:'pagesOpen',
  prevBtnId:'prevOpen', nextBtnId:'nextOpen', sizeSelId:'pageSizeOpen'
});
const pagerDeals = new TablePager({
  tableId:'tblDeals', infoId:'infoDeals', pagesId:'pagesDeals',
  prevBtnId:'prevDeals', nextBtnId:'nextDeals', sizeSelId:'pageSizeDeals'
});

// Refresh helper used by search box
function refreshPager(tableId){
  if (tableId === 'tblOpen') pagerOpen.render();
  if (tableId === 'tblDeals') pagerDeals.render();
}
</script>
</body>
</html>
