<?php
/*********************************************************
 * performance_multi.php — NEW UI, SAME DATA as account_details
 * - Uses your PDO $conn (config.php)
 * - Uses your MT5 $api (manager_initialize.php) + MTRetCode
 * - For logged-in email: fetch ALL trade_ids from liveaccount
 * - For EACH login: MT5 snapshot, open positions, closed deals
 * - From DB: deposits/withdraws (lists + sums), copyaccounts
 * - Orange/White professional UI in an accordion per account
 *********************************************************/

declare(strict_types=1);
session_start();
error_reporting(E_ALL);
ini_set('display_errors','1');

require_once __DIR__ . '/config.php';              // defines $conn (PDO) per your config
require_once __DIR__ . '/manager_initialize.php';  // defines $api, MTRetCode, etc.

// -------- Guard --------
if (empty($_SESSION['clogin'])) {
  header('Location: login.php');
  exit;
}
$loggedEmail = $_SESSION['clogin'];

// -------- Normalize PDO --------
if (!isset($conn) || !($conn instanceof PDO)) {
  die('Database connection ($conn) not found from config.php');
}
$conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

// -------- Helpers --------
function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
function num($v, int $d=2){ return number_format((float)$v, $d, '.', ','); }
function lots_from_contract($contractUnits){ return number_format(((float)$contractUnits) * 0.0001, 2, '.', ','); }
date_default_timezone_set('Asia/Kolkata');

// -------- Fetch ALL trade_ids for this user --------
$tradeIds = [];
$stmt = $conn->prepare("
  SELECT trade_id
  FROM liveaccount
  WHERE email = :eid AND trade_id IS NOT NULL AND trade_id <> ''
  ORDER BY id DESC
");
$stmt->execute([':eid'=>$loggedEmail]);
foreach ($stmt->fetchAll(PDO::FETCH_OBJ) as $r) {
  $tid = trim((string)($r->trade_id ?? ''));
  if ($tid !== '') $tradeIds[$tid] = true; // de-dup
}
$tradeIds = array_keys($tradeIds);

// -------- Per-account fetch using SAME MT5 calls as in account_details --------
$accounts = [];   // tid => [snapshot, positions, deals, sums, lists, copy]
$errors   = [];

foreach ($tradeIds as $tid) {
  $login = (int)$tid;

  // MT5: window (same defaults your page uses)
  $from   = 0;              // all history (or adjust)
  $to     = time();
  $offset = 0; $offsets = 0;

  // (1) Open positions
  $open_total = 0;
  $ret = $api->PositionGetTotal($login, $open_total);
  if ($ret != MTRetCode::MT_RET_OK) { $errors[] = "[$tid] PositionGetTotal failed, code=$ret"; continue; }

  $positions = [];
  $ret = $api->PositionGetPage($login, $offset, (int)$open_total, $positions);
  if ($ret != MTRetCode::MT_RET_OK) { $errors[] = "[$tid] PositionGetPage failed, code=$ret"; continue; }
  $resultPositions = $positions ?? [];

  // (2) Deals (history)
  $close_total = 0;
  $ret = $api->DealGetTotal($login, $from, $to, $close_total);
  if ($ret != MTRetCode::MT_RET_OK) { $errors[] = "[$tid] DealGetTotal failed, code=$ret"; continue; }

  $deals = [];
  $ret = $api->DealGetPage($login, $from, $to, $offsets, (int)$close_total, $deals);
  if ($ret != MTRetCode::MT_RET_OK) { $errors[] = "[$tid] DealGetPage failed, code=$ret"; continue; }
  $resultDeals = $deals ?? [];

  // (3) Account snapshot (once)
  $account = null;
  $ret = $api->UserAccountGet($login, $account);
  if ($ret != MTRetCode::MT_RET_OK) { $errors[] = "[$tid] UserAccountGet failed, code=$ret"; continue; }

  // Snapshot fields (as used in your page)
  $balance   = $account->Balance      ?? 0;
  $credit    = $account->Credit       ?? 0;
  $equity    = $account->Equity       ?? 0;
  $margin    = $account->Margin       ?? 0;
  $free      = $account->MarginFree   ?? 0;
  $ml        = $account->MarginLevel  ?? 0;
  $profit    = $account->Profit       ?? 0;     // floating
  $currency  = $account->Currency     ?? 'USD';
  $group     = $account->Group        ?? '';
  $leverage  = $account->Leverage     ?? '';
  $name      = $account->Name         ?? '';

  // (4) Totals from DB: deposits/withdrawals (status=1)
  $sumDeposit = 0.0; $sumWithdraw = 0.0;

  $stD = $conn->prepare("SELECT COALESCE(SUM(deposit_amount),0) AS amount
                         FROM wallet_deposit
                         WHERE email = :eid AND trade_id = :tid AND status = 1");
  $stD->execute([':eid'=>$loggedEmail, ':tid'=>$tid]);
  if ($r = $stD->fetch(PDO::FETCH_OBJ)) $sumDeposit = (float)$r->amount;

  $stW = $conn->prepare("SELECT COALESCE(SUM(withdraw_amount),0) AS amount
                         FROM wallet_withdraw
                         WHERE email = :eid AND trade_id = :tid AND status = 1");
  $stW->execute([':eid'=>$loggedEmail, ':tid'=>$tid]);
  if ($r = $stW->fetch(PDO::FETCH_OBJ)) $sumWithdraw = (float)$r->amount;

  // (5) Full lists for deposit/withdraw tables (like your page)
  $listDeposit = [];
  $st = $conn->prepare("SELECT * FROM wallet_deposit WHERE trade_id = :tid");
  $st->execute([':tid'=>$tid]);
  $listDeposit = $st->fetchAll(PDO::FETCH_OBJ);

  $listWithdraw = [];
  $st = $conn->prepare("SELECT * FROM wallet_withdraw WHERE trade_id = :tid");
  $st->execute([':tid'=>$tid]);
  $listWithdraw = $st->fetchAll(PDO::FETCH_OBJ);

  // (6) Copier table



  $accounts[$tid] = [
    'snapshot' => [
      'login'=>$tid, 'name'=>$name, 'group'=>$group, 'leverage'=>$leverage,
      'currency'=>$currency, 'balance'=>$balance, 'credit'=>$credit,
      'equity'=>$equity, 'margin'=>$margin, 'free'=>$free, 'margin_level'=>$ml,
      'profit'=>$profit
    ],
    'positions' => $resultPositions,
    'deals'     => $resultDeals,
    'sumDeposit'=> $sumDeposit,
    'sumWithdraw'=>$sumWithdraw,
    'listDeposit'=> $listDeposit,
    'listWithdraw'=> $listWithdraw,
    
  ];
}

// -------- Aggregates across accounts (optional top KPIs) --------
$agg = ['balance'=>0.0,'equity'=>0.0,'margin'=>0.0,'free'=>0.0,'profit'=>0.0,'deposit'=>0.0,'withdraw'=>0.0];
$displayCcy = 'USD';
if ($accounts) {
  $first = reset($accounts);
  $displayCcy = $first['snapshot']['currency'] ?? 'USD';
}
foreach ($accounts as $acc) {
  $s = $acc['snapshot'];
  $agg['balance']  += (float)$s['balance'];
  $agg['equity']   += (float)$s['equity'];
  $agg['margin']   += (float)$s['margin'];
  $agg['free']     += (float)$s['free'];
  $agg['profit']   += (float)$s['profit'];
  $agg['deposit']  += (float)$acc['sumDeposit'];
  $agg['withdraw'] += (float)$acc['sumWithdraw'];
}
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<title>Performance — All Accounts</title>
<meta name="viewport" content="width=device-width, initial-scale=1">

<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<title>Performance — All Accounts</title>
<meta name="viewport" content="width=device-width, initial-scale=1">

<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<style>
  :root{
    --o:#ff7a00; --o6:#ff6a00; --o7:#e55f00;
    --bg:#ffffff; --text:#1f2328; --muted:#6b7280; --soft:#fff7f0; --br:#f0f0f0;
  }
  body{
    background:#fafafa; 
    color:var(--text);
    overflow-x:hidden; /* ✅ stops right scroll */
  }
  .wrap{
    max-width:1250px; 
    margin:24px auto; 
    padding:0 12px; /* ✅ balanced padding */
  }
  .hdr{display:flex; flex-wrap:wrap; align-items:center; justify-content:space-between; gap:10px; margin-bottom:16px;}
  .ttl{margin:0; font-weight:900; letter-spacing:.2px;}
  .sub{margin:0; color:var(--muted);}
  .card{
    background:var(--bg); border:1px solid var(--br); border-radius:16px; padding:18px; height:100%;
    box-shadow:0 6px 18px rgba(0,0,0,.05);
    max-width:100%; /* ✅ prevents overflow */
  }
  .kpi .l{color:var(--muted); font-weight:600; font-size:.85rem;}
  .kpi .v{font-size:1.6rem; font-weight:800; line-height:1.15;}
  .badge-soft{background:var(--soft); color:var(--o7); border:1px dashed var(--o6); font-weight:700;}
  .sect{font-weight:800; font-size:1.05rem; margin:0 0 12px;}
  .tbl thead th{background:#fff3e8; color:#b84e00; border-bottom:2px solid var(--o6); font-size:.9rem;}
  .tbl tbody td{vertical-align:middle;}
  .text-o{color:var(--o7)!important;}
  .lnk-o{color:var(--o7); text-decoration:none; font-weight:700;}
  .lnk-o:hover{text-decoration:underline;}
  .divider{height:1px; background:var(--br); margin:16px 0;}
  .accordion .accordion-button{font-weight:800;}
  .accordion .accordion-button:not(.collapsed){background:#fff3e8; color:#b84e00;}
  .btn-o{background:var(--o); color:#fff; border:none; font-weight:700;}
  .btn-o:hover{background:var(--o7); color:#fff;}
  .pill{display:inline-flex; align-items:center; gap:8px; background:var(--soft); border:1px solid var(--br); padding:6px 10px; border-radius:999px; font-weight:700; color:var(--o7);}
  /* ✅ fix table overflow on small screens */
  .table-responsive {overflow-x:auto; -webkit-overflow-scrolling:touch; max-width:100%;}
</style>
</head>
<body>
    

     

                <!-- ✅ keep your tables inside .table-responsive as before -->
                <!-- deposit, withdraw, positions, deals remain same -->
        



<body>
     <?php include __DIR__ . '/header.php'; ?>
<div class="flex flex-col md:flex-row">
   
  <?php include __DIR__ . '/side_bar.php'; ?>

  <main class="flex-1 p-4 sm:p-6 md:ml-64">
    <div class="max-w-7xl mx-auto">
 <!-- Overall KPIs -->
      <div class="row g-2 mb-2" style="margin-top:40px;">
        <div class="col-md-2 col-6"><div class="card kpi"><div class="l">Total Balance (<?= h($displayCcy) ?>)</div><div class="v"><?= num($agg['balance']) ?></div><div class="small text-muted mt-2">Deposits: <?= num($agg['deposit']) ?></div></div></div>
        <div class="col-md-2 col-6"><div class="card kpi"><div class="l">Total Equity</div><div class="v"><?= num($agg['equity']) ?></div><div class="small text-muted mt-2">Floating P/L: <?= num($agg['profit']) ?></div></div></div>
        <div class="col-md-2 col-6"><div class="card kpi"><div class="l">Total Margin</div><div class="v"><?= num($agg['margin']) ?></div><div class="small text-muted mt-2">Free Margin: <?= num($agg['free']) ?></div></div></div>
        <div class="col-md-2 col-6"><div class="card kpi"><div class="l">Total Withdrawals</div><div class="v"><?= num($agg['withdraw']) ?></div><div class="small text-muted mt-2">Currency: <?= h($displayCcy) ?></div></div></div>
      </div>


  <div class="divider"></div>

 
      <!-- Accordion accounts (same logic you already had) -->
      <div class="accordion" id="acc">
        <?php if (!$accounts): ?>
          <div class="card"><div class="text-center text-muted py-4">No trade accounts found for your email.</div></div>
        <?php endif; ?>

        <?php $i=0; foreach ($accounts as $tid => $acc): $i++; $s = $acc['snapshot']; ?>
          <div class="accordion-item mb-3 border-0" style="max-width:100%;">
            <h2 class="accordion-header" id="h<?= $i ?>">
              <button class="accordion-button <?= $i>1?'collapsed':'' ?>" type="button" data-bs-toggle="collapse" data-bs-target="#c<?= $i ?>" aria-expanded="<?= $i==1?'true':'false' ?>" aria-controls="c<?= $i ?>">
                <span class="me-2">Login:</span> <strong class="me-3"><?= h($tid) ?></strong>
                <span class="me-3">• <?= h($s['name'] ?? '—') ?></span>
                <span class="badge bg-light text-o border me-2">Group: <?= h($s['group'] ?? '—') ?></span>
                <span class="badge bg-light text-o border">Leverage: 1:<?= h($s['leverage'] ?? '—') ?></span>
              </button>
            </h2>
            <div id="c<?= $i ?>" class="accordion-collapse collapse <?= $i==1?'show':'' ?>" aria-labelledby="h<?= $i ?>" data-bs-parent="#acc">
              <div class="accordion-body" style="max-width:100%; overflow-x:auto;">
      
            <!-- Snapshot KPIs -->
            <div class="row g-2 mb-2">
              <div class="col-md-2 col-6"><div class="card kpi"><div class="l">Balance (<?= h($s['currency'] ?? 'USD') ?>)</div><div class="v"><?= num($s['balance']) ?></div><div class="small text-muted mt-2">Credit: <?= num($s['credit']) ?></div></div></div>
              <div class="col-md-2 col-6"><div class="card kpi"><div class="l">Equity</div><div class="v"><?= num($s['equity']) ?></div><div class="small text-muted mt-2">Floating P/L: <?= num($s['profit']) ?></div></div></div>
              <div class="col-md-2 col-6"><div class="card kpi"><div class="l">Margin</div><div class="v"><?= num($s['margin']) ?></div><div class="small text-muted mt-2">Free Margin: <?= num($s['free']) ?></div></div></div>
              <div class="col-md-2 col-6"><div class="card kpi"><div class="l">Margin Level</div><div class="v"><?= $s['margin_level'] ? num($s['margin_level']) . '%' : '—' ?></div><div class="small text-muted mt-2">Leverage: 1:<?= h($s['leverage']) ?></div></div></div>
            </div>

            <!-- Deposits / Withdrawals summary -->
            <div class="row g-2 mb-2">
              <div class="col-md-5"><div class="card">
                <div class="d-flex align-items-center justify-content-between"><div class="sect">Deposits (Total Approved)</div><span class="pill">Sum: <?= num($acc['sumDeposit']) ?></span></div>
                <div class="table-responsive mt-3">
                  <table class="table table-hover align-middle tbl">
                    <thead><tr><th>Trading Account</th><th>Deposit Type</th><th>Amount</th><th>Status</th><th>Remarks</th></tr></thead>
                    <tbody>
                      <?php if ($acc['listDeposit']): foreach ($acc['listDeposit'] as $row): ?>
                        <tr>
                          <td><?= h($row->trade_id ?? '') ?></td>
                         
                          <td><?= h($row->deposit_type ?? '') ?></td>
                          <td class="text-end"><?= num($row->deposit_amount ?? 0) ?></td>
                          <td><?= h($row->Status ?? '') ?></td>
                          <td><?= h($row->AdminRemark ?? '') ?></td>
                          
                        </tr>
                      <?php endforeach; else: ?>
                        <tr><td colspan="7" class="text-center text-muted">No deposits.</td></tr>
                      <?php endif; ?>
                    </tbody>
                  </table>
                </div>
              </div></div>
              <div class="col-md-5"><div class="card">
                <div class="d-flex align-items-center justify-content-between"><div class="sect">Withdrawals (Total Approved)</div><span class="pill">Sum: <?= num($acc['sumWithdraw']) ?></span></div>
                <div class="table-responsive mt-3">
                  <table class="table table-hover align-middle tbl">
                    <thead><tr><th>Trading Account</th><th>Withdraw Type</th><th>Amount</th><th>Status</th><th>Remarks</th></tr></thead>
                    <tbody>
                      <?php if ($acc['listWithdraw']): foreach ($acc['listWithdraw'] as $row): ?>
                        <tr>
                          <td><?= h($row->trade_id ?? '') ?></td>
                          <td><?= h($row->withdraw_type ?? '') ?></td>
                          <td class="text-end"><?= num($row->withdraw_amount ?? 0) ?></td>
                          <td><?= h($row->Status ?? '') ?></td>
                          <td><?= h($row->AdminRemark ?? '') ?></td>
                          
                        </tr>
                      <?php endforeach; else: ?>
                        <tr><td colspan="7" class="text-center text-muted">No withdrawals.</td></tr>
                      <?php endif; ?>
                    </tbody>
                  </table>
                </div>
              </div></div>
            </div>

            <!-- Open Positions (same fields as your page) -->
            <div class="card mb-3">
              <div class="d-flex align-items-center justify-content-between mb-2">
                <div class="sect">Open Positions</div><a class="lnk-o" href="?r=<?= time() ?>">Refresh</a>
              </div>
              <div class="table-responsive">
                <table class="table table-hover align-middle tbl">
                  <thead>
                    <tr>
                      <th>Symbol</th><th>Open</th><th>Current</th><th>SL</th><th>TP</th>
                      <th class="text-end">Volume</th><th>Action</th><th class="text-end">P/L</th><th class="text-end">Swap</th><th>Placed Time</th>
                    </tr>
                  </thead>
                  <tbody>
                    <?php if ($acc['positions']): foreach ($acc['positions'] as $p): ?>
                      <tr>
                        <td><?= h($p->Symbol) ?></td>
                        <td class="text-end"><?= h($p->PriceOpen) ?></td>
                        <td class="text-end"><?= h($p->PriceCurrent) ?></td>
                        <td class="text-end"><?= h($p->PriceSL) ?></td>
                        <td class="text-end"><?= h($p->PriceTP) ?></td>
                        <td class="text-end"><?= lots_from_contract($p->Volume ?? 0) ?></td>
                        <td><?= (($p->Action ?? 0) == 1) ? 'Sell' : 'Buy' ?></td>
                        <td class="text-end"><?= num($p->Profit ?? 0) ?></td>
                        <td class="text-end"><?= num($p->Storage ?? 0) ?></td>
                        <td>
                          <?php
                            $ts = (int)($p->TimeCreate ?? 0);
                            echo $ts ? (new DateTime("@".$ts))->setTimezone(new DateTimeZone('Asia/Kolkata'))->format('F j, Y, h:i:s a') : '-';
                          ?>
                        </td>
                      </tr>
                    <?php endforeach; else: ?>
                      <tr><td colspan="10" class="text-center text-muted py-4">No open positions.</td></tr>
                    <?php endif; ?>
                  </tbody>
                </table>
              </div>
            </div>

            <!-- Closed Deals / History (same filters/fields) -->
            <div class="card">
              <div class="sect">Recent Closed Orders</div>
              <div class="table-responsive">
                <table class="table table-hover align-middle tbl">
                  <thead>
                    <tr>
                      <th>Open Time</th><th>Ticket</th><th>Action</th><th class="text-end">Volume</th>
                      <th>Symbol</th><th class="text-end">Open Price</th><th class="text-end">Close Price</th>
                      <th>Close Time</th><th class="text-end">Commission</th><th class="text-end">Swap</th><th class="text-end">Profit</th>
                    </tr>
                  </thead>
                  <tbody>
                    <?php
                      $hadRows = false;
                      foreach ($acc['deals'] as $t):
                        $lots = (float)($t->VolumeClosed ?? 0) * 0.0001;
                        if (($t->Order ?? 0) == 0 || $lots == 0.0) continue; // same filter as your page
                        $hadRows = true;

                        $openTime = !empty($t->Time)
                          ? (new DateTime("@".$t->Time))->setTimezone(new DateTimeZone('Asia/Kolkata'))->format('l, F j, Y g:i A')
                          : '-';
                        $closeBase = !empty($t->TimeMsc) ? (int)floor(((int)$t->TimeMsc)/1000) : 0;
                        $closeMs   = !empty($t->TimeMsc) ? ((int)$t->TimeMsc % 1000) : 0;
                        $closeDT   = $closeBase
                          ? (new DateTime("@".$closeBase))->setTimezone(new DateTimeZone('Asia/Kolkata'))->format('l, F j, Y g:i A') . '.' . $closeMs
                          : '-';
                    ?>
                      <tr>
                        <td><?= h($openTime) ?></td>
                        <td><?= h($t->Order) ?></td>
                        <td><?= (($t->Action ?? 0) == 1) ? 'Sell' : 'Buy' ?></td>
                        <td class="text-end"><?= number_format($lots, 2) ?></td>
                        <td><?= h($t->Symbol) ?></td>
                        <td class="text-end"><?= h($t->PricePosition) ?></td>
                        <td class="text-end"><?= h($t->Price) ?></td>
                        <td><?= h($closeDT) ?></td>
                        <td class="text-end"><?= num($t->Commission ?? 0) ?></td>
                        <td class="text-end"><?= num($t->Storage ?? 0) ?></td>
                        <td class="text-end"><?= num($t->Profit ?? 0) ?></td>
                      </tr>
                    <?php endforeach; ?>
                    <?php if (!$hadRows): ?>
                      <tr><td colspan="11" class="text-center text-muted py-4">No closed orders found.</td></tr>
                    <?php endif; ?>
                  </tbody>
                </table>
              </div>
            </div>

          

          </div><!-- /accordion-body -->
        </div><!-- /collapse -->
      </div><!-- /item -->
    <?php endforeach; ?>
  </div><!-- /accordion -->

</div><!-- /wrap -->

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
