<?php
session_start();
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once "config.php"; // must set $conn = new PDO(...)

if (empty($_SESSION['clogin'])) {
  echo "<script>window.location.href='login.php';</script>";
  exit;
}

$eid = $_SESSION['clogin']; // Get logged-in user's email

/* ---------------- Inputs ---------------- */
$from       = isset($_GET['from']) ? trim($_GET['from']) : '';  // yyyy-mm-dd
$to         = isset($_GET['to'])   ? trim($_GET['to'])   : '';
$q          = isset($_GET['q'])    ? trim($_GET['q'])    : '';
$per_page   = isset($_GET['per_page']) ? max(5, (int)$_GET['per_page']) : 10;
$page       = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;

/* ---------------- WHERE building ---------------- */
$where  = ['email = :email'];  // Only get the current user's email
$params = [':email' => $eid];

if ($from !== '' && $to !== '') {
  $where[] = "DATE(withdraw_date) BETWEEN :from AND :to";
  $params[':from'] = $from;
  $params[':to']   = $to;
} elseif ($from !== '') {
  $where[] = "DATE(withdraw_date) >= :from";
  $params[':from'] = $from;
} elseif ($to !== '') {
  $where[] = "DATE(withdraw_date) <= :to";
  $params[':to'] = $to;
}

if ($q !== '') {
  $where[] = "(LOWER(email) LIKE :q OR LOWER(trade_id) LIKE :q OR LOWER(withdraw_type) LIKE :q OR CAST(withdraw_amount AS CHAR) LIKE :q)";
  $params[':q'] = '%'.strtolower($q).'%';
}

$whereSql = 'WHERE ' . implode(' AND ', $where);

/* ---------------- Count ---------------- */
$sqlCount = "SELECT COUNT(*) FROM wallet_withdraw $whereSql";
$stmt = $conn->prepare($sqlCount);
foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
$stmt->execute();
$total = (int)$stmt->fetchColumn();

$total_pages = max(1, (int)ceil($total / $per_page));
$page   = min($page, $total_pages);
$offset = ($page - 1) * $per_page;

/* ---------------- Page rows ---------------- */
$sql = "SELECT id,email,trade_id,withdraw_amount,withdraw_type,Status,withdraw_date
        FROM wallet_withdraw
        $whereSql
        ORDER BY withdraw_date DESC, id DESC
        LIMIT :limit OFFSET :offset";
$stmt = $conn->prepare($sql);
foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
$stmt->bindValue(':limit',  $per_page, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset,   PDO::PARAM_INT);
$stmt->execute();
$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* ---------------- Helpers ---------------- */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function status_badge($s){
  if ($s == 0) return '<span class="text-yellow-600 font-semibold">Pending</span>';
  if ($s == 1) return '<span class="text-green-600 font-semibold">Approved</span>';
  if ($s == 2) return '<span class="text-red-600 font-semibold">Rejected</span>';
  return 'Unknown';
}
function qs($overrides=[]){
  $base = $_GET;
  foreach ($overrides as $k=>$v) $base[$k] = $v;
  return http_build_query($base);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
  <title>Withdrawal Report</title>
  <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
</head>
<body class="bg-gray-50">
<?php include "header.php"; ?>
<div class="flex flex-col md:flex-row min-h-screen">
  <?php include "side_bar.php"; ?>

  <main class="flex-1 p-6 md:ml-64">
    <div class="max-w-7xl mx-auto">
      <h1 class="text-3xl md:text-4xl font-bold text-center mb-6 text-yellow-700">Withdrawal Report</h1>

      <!-- Filter bar (like your screenshot) -->
      <form method="get" class="bg-white border border-gray-200 rounded-xl p-4 md:p-5 mb-5 shadow-sm">
        <div class="grid grid-cols-1 md:grid-cols-5 gap-4 items-end">
          <div class="md:col-span-1">
            <label class="block text-gray-700 font-medium mb-1">From date</label>
            <input type="date" name="from" id="from" value="<?=h($from)?>"
                   class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-yellow-600 focus:border-yellow-600"
                   placeholder="dd-mm-yyyy">
          </div>

          <div class="md:col-span-1">
            <label class="block text-gray-700 font-medium mb-1">To date</label>
            <input type="date" name="to" id="to" value="<?=h($to)?>"
                   class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-yellow-600 focus:border-yellow-600"
                   placeholder="dd-mm-yyyy">
          </div>

          <div class="md:col-span-2">
            <label class="block text-gray-700 font-medium mb-1">Live search (Trade ID / Method )</label>
            <input type="text" id="live" value="" placeholder="Type to filter..."
                   class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-yellow-600 focus:border-yellow-600">
            <!-- Optional server-side search box (press Enter to submit to server)
                 If you want only client-side live search, keep q hidden. -->
            <input type="hidden" name="q" id="q" value="<?=h($q)?>">
          </div>

          <div class="md:col-span-1 flex items-end justify-between md:justify-end gap-4">
            <a href="?reset=1" class="text-gray-500 hover:text-gray-700">Reset</a>
            <button class="bg-yellow-600 hover:bg-yellow-700 text-white font-semibold px-5 py-2 rounded-lg shadow">
              Apply Date Filter
            </button>
          </div>
        </div>
      </form>

      <!-- Table -->
      <div class="overflow-x-auto">
        <table class="min-w-full bg-white border border-gray-200 rounded-lg shadow-md" id="reportTable">
          <thead>
          <tr class="bg-gray-100">
            <th class="px-3 py-2 border-b text-left text-yellow-700">Email</th>
            <th class="px-3 py-2 border-b text-left text-yellow-700">MT5 ID</th>
            <th class="px-3 py-2 border-b text-left text-yellow-700">Amount</th>
            <th class="px-3 py-2 border-b text-left text-yellow-700">Payment Method</th>
            <th class="px-3 py-2 border-b text-left text-yellow-700">Status</th>
            <th class="px-3 py-2 border-b text-left text-yellow-700">Date</th>
          </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="6" class="px-3 py-6 text-center text-gray-500">No withdrawals found.</td></tr>
          <?php endif; ?>

          <?php foreach ($rows as $withdrawal): ?>
            <tr class="hover:bg-gray-50"
                data-rowtext="<?=h(strtolower($withdrawal['email'].' '.$withdrawal['trade_id'].' '.$withdrawal['withdraw_type'].' '.$withdrawal['withdraw_amount']))?>"
                data-date="<?=h(substr($withdrawal['withdraw_date'],0,10))?>">
              <td class="px-3 py-3 border-b"><?=h($withdrawal['email'])?></td>
              <td class="px-3 py-3 border-b"><?=h($withdrawal['trade_id'])?></td>
              <td class="px-3 py-3 border-b"><?= '$'.number_format((float)$withdrawal['withdraw_amount'], 2)?></td>
              <td class="px-3 py-3 border-b"><?=h($withdrawal['withdraw_type'])?></td>
              <td class="px-3 py-3 border-b"><?= status_badge($withdrawal['Status']) ?></td>
              <td class="px-3 py-3 border-b"><?=h($withdrawal['withdraw_date'])?></td>
            </tr>
          <?php endforeach; ?>
          </tbody>
        </table>
      </div>

      <!-- Pagination -->
      <div class="mt-4 flex items-center justify-between">
        <div class="text-sm text-gray-600">
          Showing <span class="font-semibold"><?= $total ? ($offset + 1) : 0 ?></span> to
          <span class="font-semibold"><?= $offset + count($rows) ?></span> of
          <span class="font-semibold"><?= $total ?></span> entries
        </div>
        <div class="flex gap-2">
          <?php if ($page > 1): ?>
            <a class="px-3 py-2 border rounded hover:bg-gray-50"
               href="?<?=qs(['page'=>$page-1])?>">Prev</a>
          <?php endif; ?>
          <?php if ($page < $total_pages): ?>
            <a class="px-3 py-2 border rounded hover:bg-gray-50"
               href="?<?=qs(['page'=>$page+1])?>">Next</a>
          <?php endif; ?>
        </div>
      </div>

      <?php include "footer.php"; ?>
    </div>
  </main>
</div>

<script>
/* ---- Client-side live search + date range (visual) ---- */
const live = document.getElementById('live');
const from = document.getElementById('from');
const to   = document.getElementById('to');
const rows = Array.from(document.querySelectorAll('#reportTable tbody tr'));

function inRange(dateStr, fromStr, toStr){
  if (!fromStr && !toStr) return true;
  const d  = dateStr ? new Date(dateStr) : null;
  if (!d) return false;
  if (fromStr && d < new Date(fromStr)) return false;
  if (toStr   && d > new Date(toStr))   return false;
  return true;
}
function applyLive(){
  const q = (live.value || '').toLowerCase();
  const f = from.value || '';
  const t = to.value   || '';
  rows.forEach(tr=>{
    const txt  = tr.dataset.rowtext || '';
    const d    = tr.dataset.date || '';
    const okQ  = (q === '' || txt.includes(q));
    const okDt = inRange(d, f, t);
    tr.style.display = (okQ && okDt) ? '' : 'none';
  });
}
[live, from, to].forEach(el => el && el.addEventListener('input', applyLive));
applyLive();

/* Optional: when user presses Enter in live search, also set the hidden q for server search */
live.addEventListener('keydown', e=>{
  if(e.key === 'Enter'){
    document.getElementById('q').value = live.value;
    e.target.form.submit();
  }
});
</script>
</body>
</html>
